package com.xebialabs.deployit.plugin.generic.deployed;

import com.google.common.base.Strings;
import com.xebialabs.deployit.plugin.api.deployment.planning.*;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.DeployableArtifact;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.generic.step.ScriptExecutionStep;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.local.LocalFile;

import java.util.Set;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Sets.newHashSet;

@SuppressWarnings("serial")
@Metadata(virtual = true, description = "A script executed on a generic container")
public class ExecutedScript<D extends Deployable> extends AbstractDeployed<D> {

     @Property(required = true, hidden=true, description = "Classpath to the script that is uploaded and executed on the generic container for the create operation.")
    private String createScript;

    @Property(required = false, hidden=true, description = "Classpath to the script that is uploaded and executed on the generic container for the modify operation.")
    private String modifyScript;

    @Property(required = false, hidden=true, description = "Classpath to the script that is uploaded and executed on the generic container for the destroy operation.")
    private String destroyScript;

    @Property(required = false, hidden=true, description = "Classpath to the script that is uploaded and executed on the generic container for the noop operation.")
    private String noopScript;

    @Property(required = false, hidden=true, description = "Name of working directory on target host. Default is to create a temporary directory which is deleted when connection is closed.")
    private String remoteWorkingDirectoryPath;

    @Property(required = false, hidden=true, defaultValue = "false", description = "Retain the specified working directory on target host after completion.")
    private boolean retainRemoteWorkingDirectory = false;

    @Property(hidden = true, required=false, description = "Additional classpath resources that should be uploaded to the working directory before executing the script.")
    private Set<String> classpathResources = newHashSet();

    @Property(hidden = true, required=false, description = "Additional template classpath resources that should be uploaded to the working directory before executing the script." +
            "The template is first rendered and the rendered content copied to a file, with the same name as the template, in the working directory.")
    private Set<String> templateClasspathResources = newHashSet();

    protected boolean addStep(DeploymentPlanningContext ctx, int order, String script, String verb) {
        if (!Strings.nullToEmpty(script).trim().isEmpty()) {
            ScriptExecutionStep step = new ScriptExecutionStep(order, script, getContainer(), getDeployedAsFreeMarkerContext(), getDescription(verb));
            LocalFile file = (LocalFile)getArtifactIfPresent();
            if (file != null) {
                step.setArtifact(file.getFile());
            }

            if (!Strings.isNullOrEmpty(getRemoteWorkingDirectoryPath())) {
                step.setRemoteWorkingDirPath(getRemoteWorkingDirectoryPath());
                step.setRetainRemoteWorkingDirOnCompletion(isRetainRemoteWorkingDirectory());
            }

            step.setTemplateClasspathResources(newArrayList(getTemplateClasspathResources()));
            step.setClasspathResources(newArrayList(getClasspathResources()));
            ctx.addStep(step);
            return true;
        } else {
            return false;
        }
    }

    protected OverthereFile getArtifactIfPresent() {
        if (getDeployable() instanceof DeployableArtifact) {
            return ((DeployableArtifact)getDeployable()).getFile();
        }
        return null;
    }

    @SuppressWarnings("unchecked")
    @Create
    public void executeCreate(DeploymentPlanningContext ctx, Delta d) {
	    ExecutedScript<D> which = (ExecutedScript<D>) d.getDeployed();
	    which._create(ctx);
    }

	private void _create(DeploymentPlanningContext ctx) {
		addStep(ctx, getCreateOrder(), getCreateScript(), getCreateVerb());
	}

	@Modify
    public void executeModify(DeploymentPlanningContext ctx, Delta d) {
        boolean modifyStepAdded = addStep(ctx, getModifyOrder(), getModifyScript(), getModifyVerb());
        if (!modifyStepAdded) {
            executeDestroy(ctx, d);
            executeCreate(ctx, d);
        }
    }

    @SuppressWarnings("unchecked")
    @Destroy
    public void executeDestroy(DeploymentPlanningContext ctx, Delta d) {
	    ExecutedScript<D> which = (ExecutedScript<D>) d.getPrevious();
	    which._destroy(ctx);
    }

    private void _destroy(DeploymentPlanningContext ctx) {
        addStep(ctx, getDestroyOrder(), getDestroyScript(), getDestroyVerb());
    }

    @Noop
    public void executeNoop(DeploymentPlanningContext ctx) {
        addStep(ctx, getNoopOrder(), getNoopScript(), getNoopVerb());
    }

    public String getCreateScript() {
        return resolveExpression(createScript);
    }

    public void setCreateScript(String createScript) {
        this.createScript = createScript;
    }

    public String getModifyScript() {
        return resolveExpression(modifyScript);
    }

    public void setModifyScript(String modifyScript) {
        this.modifyScript = modifyScript;
    }

    public String getDestroyScript() {
        return resolveExpression(destroyScript);
    }

    public void setDestroyScript(String destroyScript) {
        this.destroyScript = destroyScript;
    }

    public String getNoopScript() {
        return resolveExpression(noopScript);
    }

    public void setNoopScript(String noopScript) {
        this.noopScript = noopScript;
    }

    public String getRemoteWorkingDirectoryPath() {
        return resolveExpression(remoteWorkingDirectoryPath);
    }

    public void setRemoteWorkingDirectoryPath(String remoteWorkingDirectoryPath) {
        this.remoteWorkingDirectoryPath = remoteWorkingDirectoryPath;
    }

    public boolean isRetainRemoteWorkingDirectory() {
        return retainRemoteWorkingDirectory;
    }

    public void setRetainRemoteWorkingDirectory(boolean retainRemoteWorkingDirectory) {
        this.retainRemoteWorkingDirectory = retainRemoteWorkingDirectory;
    }

    public Set<String> getClasspathResources() {
        return resolveExpression(classpathResources);
    }

    public void setClasspathResources(Set<String> classpathResources) {
        this.classpathResources = classpathResources;
    }

    public Set<String> getTemplateClasspathResources() {
        return resolveExpression(templateClasspathResources);
    }

    public void setTemplateClasspathResources(Set<String> templateClasspathResources) {
        this.templateClasspathResources = templateClasspathResources;
    }
}
