package com.xebialabs.deployit.plugin.generic.step;

import com.google.common.base.Preconditions;
import com.xebialabs.deployit.plugin.overthere.ExecutionContextOverthereProcessOutputHandler;
import com.xebialabs.deployit.plugin.overthere.HostContainer;
import com.xebialabs.overthere.CmdLine;
import com.xebialabs.overthere.OperatingSystemFamily;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.OverthereProcessOutputHandler;
import org.slf4j.MDC;

import java.io.File;
import java.util.List;
import java.util.Map;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;

@SuppressWarnings("serial")
public class ScriptExecutionStep extends GenericBaseStep {

    private static final String MDC_KEY_SCRIPT_PATH = "scriptPath";

    private String scriptTemplatePath;
    private Map<String, Object> vars;

    private File artifact;

    private List<File> fileResources = newArrayList();
    private List<String> classpathResources = newArrayList();
    private List<String> templateClasspathResources = newArrayList();

    public ScriptExecutionStep(int order, String scriptPath, HostContainer container, Map<String, Object> vars, String description) {
        super(order, description, container);
        this.scriptTemplatePath = scriptPath;
        this.vars = newHashMap(vars);
        this.vars.put("step",this);
        Preconditions.checkNotNull(scriptTemplatePath);
    }

    @Override
    public Result doExecute() throws Exception {
        MDC.put(MDC_KEY_SCRIPT_PATH, scriptTemplatePath);
        try {
            uploadArtifactIfPresent();
            uploadFileResources();
            uploadClasspathResources();
            uploadTemplateClasspathResources();
            OverthereFile executable = uploadScriptToExecute(scriptTemplatePath);
            setWorkingDirectory();
            return executeScript(executable);
        } finally {
            MDC.remove(MDC_KEY_SCRIPT_PATH);
        }
    }

    protected OverthereFile uploadArtifactIfPresent() {
        if (artifact != null) {
            return uploadToWorkingDirectory(artifact, artifact.getName());
        }
        return null;
    }

    protected void uploadFileResources() {
        for (File fileResource : fileResources) {
            String type = fileResource.isDirectory() ? "directory" : "file";
            getCtx().logOutput("Uploading " + type + " " + fileResource.getName() + " to working directory.");
            uploadToWorkingDirectory(fileResource, fileResource.getName());
        }
    }

    protected void uploadClasspathResources() {
        for (String cpResource : classpathResources) {
            getCtx().logOutput("Uploading classpath resource " + cpResource + " to working directory.");
            uploadResourceToWorkingDirectory(cpResource);
        }
    }

    protected void uploadTemplateClasspathResources() {
        for (String cpTemplate : templateClasspathResources) {
            uploadTemplateResourceToWorkingDirectory(cpTemplate);
        }
    }

    public OverthereFile uploadTemplateResourceToWorkingDirectory(String template) {
        String osSpecificTemplate = resolveOsSpecificTemplate(template);
        String content = evaluateTemplate(osSpecificTemplate, vars);
        logger.debug("Generated content for template {} :\n {} ", template, content);
        String fileName = resolveOsSpecificFileName(osSpecificTemplate);
        logger.debug("Uploading file " + fileName + " to working directory.");
        getCtx().logOutput("Uploading file " + fileName + " to working directory.");
        return uploadToWorkingDirectory(content, fileName);
    }

    protected void setWorkingDirectory() {
    	getRemoteConnection().setWorkingDirectory(getRemoteWorkingDirectory());
    }

	protected OverthereFile uploadScriptToExecute(String scriptTemplatePath) {
        OverthereFile executable = uploadTemplateResourceToWorkingDirectory(scriptTemplatePath);
        executable.setExecutable(true);
        return executable;
    }

    protected Result executeScript(OverthereFile executable) {
        CmdLine cmdLine = CmdLine.build(executable.getPath());
        OverthereProcessOutputHandler handle = new ExecutionContextOverthereProcessOutputHandler(getCtx());
        getCtx().logOutput("Executing " + executable.getPath() + " on host " + getContainer().getHost());
        int rc = getRemoteConnection().execute(handle, cmdLine);
        if (rc != 0) {
            getCtx().logError("Execution failed with return code " + rc);
            return Result.Fail;
        }
        return Result.Success;
    }

    public List<File> getFileResources() {
        return fileResources;
    }

    public void setFileResources(List<File> fileResources) {
        this.fileResources = fileResources;
    }

    public List<String> getClasspathResources() {
        return classpathResources;
    }

    public void setClasspathResources(List<String> classpathResources) {
        this.classpathResources = classpathResources;
    }

    public List<String> getTemplateClasspathResources() {
        return templateClasspathResources;
    }

    public void setTemplateClasspathResources(List<String> templateClasspathResources) {
        this.templateClasspathResources = templateClasspathResources;
    }

    public File getArtifact() {
        return artifact;
    }

    public void setArtifact(File artifact) {
        this.artifact = artifact;
    }

    public String getUploadedArtifactPath() {
        if (artifact != null) {
            return getRemoteWorkingDirectory().getPath() + getContainer().getHost().getOs().getFileSeparator() + artifact.getName();
        }
        return null;
    }

    protected String resolveOsSpecificFileName(String template) {
         String osSpecificFile = substringAfterLast(template, '/', template);
        if (osSpecificFile.endsWith(FREEMARKER_FILE_EXT))
            osSpecificFile = osSpecificFile.substring(0, osSpecificFile.lastIndexOf(FREEMARKER_FILE_EXT));

        String fileExt = substringAfterLast(osSpecificFile, '.');
        if (fileExt == null) {
            OperatingSystemFamily os = getContainer().getHost().getOs();
            osSpecificFile = osSpecificFile + os.getScriptExtension();
        }

        return osSpecificFile;
    }



}
