package com.xebialabs.deployit.plugin.generic.freemarker;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Maps.transformValues;
import static com.google.common.collect.Sets.newHashSet;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Function;
import com.google.common.collect.Collections2;
import com.xebialabs.overthere.RuntimeIOException;

import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateException;

public class ConfigurationHolder {

	private static final AtomicReference<Configuration> holder = new AtomicReference<Configuration>();
	private static final String EXPRESSION_TOKEN = "${";
	private static final String EXPRESSION_IF_TOKEN = "<#if";

	public static Configuration getConfiguration() {
		Configuration conf = holder.get();
		if (conf == null) {
			conf = init();
			if (!holder.compareAndSet(null, conf)) {
				// another thread has initialized the reference
				conf = holder.get();
			}
		}
		return conf;
	}

	public static List<String> resolveExpression(List<String> expressions, final Object context) {
        if (expressions == null) {
            expressions = newArrayList();
        }
		return newArrayList(transformCollectionByResolvingExpression(expressions, context));
	}
	
	public static Set<String> resolveExpression(Set<String> expressions, final Object context) {
        if (expressions == null) {
            expressions = newHashSet();
        }
		return newHashSet(transformCollectionByResolvingExpression(expressions, context));
	}

	private static Collection<String> transformCollectionByResolvingExpression(Collection<String> expressions, final Object context) {
		return Collections2.transform(expressions, new Function<String, String>() {
			@Override
			public String apply(String input) {
				return resolveExpression(input, context);
			}
		});
	}

	public static Map<String, String> resolveExpression(Map<String, String> expressions, Map<String, Object> context) {
        if (expressions == null) {
            expressions = newHashMap();
        }
		final Map<String, Object> contextWithExpressions = newHashMap(context);
		contextWithExpressions.putAll(expressions);
		return transformValues(expressions, new Function<String, String>() {
			@Override
			public String apply(String input) {
				String result = input;
				while (isResolvable(result)) {
					result = resolveExpression(result, contextWithExpressions);
				}
				return result;
			}
		});
	}

	public static String resolveExpression(String expression, Object context) {
		if (!isResolvable(expression)) {
			return expression;
		}

        Configuration cfg = ConfigurationHolder.getConfiguration();
        try {
            logger.trace("Resolving expression {}.", expression);
            Template template = new Template("expression", new StringReader(expression), cfg);
            StringWriter sw = new StringWriter();
            template.process(context, sw);
            return sw.toString();
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        } catch (TemplateException e) {
            throw new RuntimeException(e);
        }
    }

	private static boolean isResolvable(String expression) {
		return (expression != null && (expression.contains(EXPRESSION_TOKEN) || expression.contains(EXPRESSION_IF_TOKEN)));
	}

    private static Configuration init() {
        Configuration conf = new Configuration();
        conf.setNumberFormat("computer");
        conf.setTemplateLoader(new ClasspathTemplateLoader());
        conf.setObjectWrapper(new CiAwareObjectWrapper());
        return conf;
    }

    private static final Logger logger = LoggerFactory.getLogger(ConfigurationHolder.class);
}
