package com.xebialabs.deployit.plugin.generic.container;

import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import com.xebialabs.deployit.plugin.api.deployment.planning.Contributor;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.generic.ci.Container;
import com.xebialabs.deployit.plugin.generic.ci.NestedContainer;
import com.xebialabs.deployit.plugin.generic.deployed.AbstractDeployed;
import com.xebialabs.deployit.plugin.generic.step.ScriptExecutionStep;
import com.xebialabs.deployit.plugin.generic.step.WaitStep;
import com.xebialabs.deployit.plugin.overthere.HostContainer;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Strings.nullToEmpty;

public class LifeCycleContributor {

    @Contributor
    public void restartContainers(Deltas deltas, DeploymentPlanningContext plan) {
        Set<Container> containers = gatherTargets(deltas.getDeltas());
        for (Container container : containers) {
            if (!nullToEmpty(container.getRestartScript()).trim().isEmpty()) {
                addRestartSteps(plan, container);
            } else {
                addStartAndStopSteps(plan, container);
            }
        }
    }

    private static void addStartAndStopSteps(DeploymentPlanningContext plan, Container target) {
        checkArgument(!nullToEmpty(target.getStartScript()).trim().isEmpty(), "start script must be specified when no restart script defined.");
        checkArgument(!nullToEmpty(target.getStopScript()).trim().isEmpty(), "stop script must be specified when no restart script defined.");

        plan.addStep(createScriptStep("Start", target, target.getStartScript(), target.getStartOrder()));
        addWaitStep(plan, "start", target, target.getStartWaitTime(), target.getStartOrder());

        plan.addStep(createScriptStep("Stop", target, target.getStopScript(), target.getStopOrder()));
        addWaitStep(plan, "stop", target, target.getStopWaitTime(), target.getStopOrder());
    }

    private static void addRestartSteps(DeploymentPlanningContext plan, Container target) {
        Step step = createScriptStep("Restart", target, target.getRestartScript(), target.getRestartOrder());
        plan.addStep(step);
        addWaitStep(plan, "restart", target, target.getRestartWaitTime(), target.getRestartOrder());
    }

    private static void addWaitStep(DeploymentPlanningContext plan, String action, Container target, int waitTime, int scriptOrder) {
        if (waitTime > 0) {
            plan.addStep(new WaitStep(scriptOrder + 1, waitTime, target.getName(), action));
        }
    }

    private static Step createScriptStep(String verb, Container container, String script, int scriptOrder) {
        Map<String, Object> vars = Maps.newHashMap();
        vars.put("container", container);
        return new ScriptExecutionStep(scriptOrder,script, container, vars, getDescription(verb, container));
    }

    private static String getDescription(String verb, Container container) {
        return String.format("%s %s", verb, container.getName());
    }


    private static Set<Container> gatherTargets(List<Delta> operations) {
        final Set<Container> targets = Sets.newTreeSet();
        for (Delta operation : operations) {
            addTarget(targets, operation.getOperation(),  operation.getDeployed());
            addTarget(targets, operation.getOperation(), operation.getPrevious());
        }
        return targets;
    }

    private static void addTarget(Set<Container> targets, Operation operation, Deployed<?, ?> deployed) {
        if (deployed == null) {
            return;
        }

        if (deployed instanceof AbstractDeployed) {
            AbstractDeployed<?> abstractDeployed = (AbstractDeployed<?>) deployed;
            boolean restartRequired = abstractDeployed.isRestartRequired();
            if (restartRequired) {
                if (operation != Operation.NOOP) {
                    targets.add(getRootContainer((HostContainer) deployed.getContainer()));
                } else if (abstractDeployed.isRestartRequiredForNoop()) {
                    targets.add(getRootContainer((HostContainer) deployed.getContainer()));
                }
            }
        }
    }

    private static Container getRootContainer(HostContainer hostContainer) {
        if (hostContainer instanceof Container) {
            return (Container)hostContainer;
        } else if (hostContainer instanceof NestedContainer) {
            return ((NestedContainer)hostContainer).getRootContainer();
        } else {
            throw new IllegalStateException("The root container for a nested container not found. NestedContains should be rooted to a generic Container.");
        }
    }
}
