package com.xebialabs.deployit.plugin.generic.step;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.StringWriter;
import java.net.URL;
import java.util.Map;

import javax.xml.parsers.ParserConfigurationException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.google.common.io.ByteStreams;
import com.google.common.io.Resources;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.generic.freemarker.CiAwareObjectWrapper;
import com.xebialabs.deployit.plugin.generic.freemarker.ConfigurationHolder;
import com.xebialabs.deployit.plugin.generic.freemarker.FileUploader;
import com.xebialabs.deployit.plugin.overthere.HostContainer;
import com.xebialabs.overthere.OperatingSystemFamily;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.RuntimeIOException;
import com.xebialabs.overthere.local.LocalConnection;
import com.xebialabs.overthere.local.LocalFile;
import com.xebialabs.overthere.util.OverthereUtils;

import freemarker.ext.dom.NodeModel;
import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateException;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Strings.emptyToNull;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.io.Closeables.closeQuietly;

@SuppressWarnings("serial")
public abstract class BaseStep implements Step {
    public static final String FREEMARKER_FILE_EXT = ".ftl";

    private String description;
    protected HostContainer container;

    private String remoteWorkingDirPath;
    private boolean retainRemoteWorkingDirOnCompletion;

    private transient OverthereConnection localConn;
    protected transient OverthereConnection remoteConn;
    protected transient ExecutionContext ctx;
    private transient OverthereFile remoteWorkingDir;

    protected BaseStep(String description, HostContainer container) {
        this.description = description;
        this.container = container;
    }

    protected BaseStep(HostContainer container) {
        this(null, container);
    }

    @Override
    public String getDescription() {
        return description;
    }

    public StepExitCode handleExecute(ExecutionContext ctx) throws Exception {
        try {
            this.ctx = ctx;
            return doExecute();
        } finally {
            disconnect();
        }
    }

    protected abstract StepExitCode doExecute() throws Exception;

    protected ExecutionContext getCtx() {
        return ctx;
    }

    public HostContainer getContainer() {
        return container;
    }

    public OverthereFile getRemoteWorkingDirectory() {
        if (remoteWorkingDir == null) {
            OverthereFile tempDir;
            if (Strings.isNullOrEmpty(getRemoteWorkingDirPath())) {
                tempDir = getRemoteConnection().getTempFile("generic_plugin", ".tmp");
            } else {
                tempDir = getRemoteConnection().getFile(getRemoteWorkingDirPath());
            }
            tempDir.mkdir();
            remoteWorkingDir = tempDir;
        }
        return remoteWorkingDir;
    }


    public OverthereConnection getLocalConnection() {
        if (localConn == null) {
            localConn = LocalConnection.getLocalConnection();
        }
        return localConn;
    }

    public OverthereConnection getRemoteConnection() {
        if (remoteConn == null) {
            remoteConn = container.getHost().getConnection();
        }
        return remoteConn;
    }

    protected void disconnect() {
        if (localConn != null)
            closeQuietly(localConn);

        if (!Strings.isNullOrEmpty(getRemoteWorkingDirPath()) && !isRetainRemoteWorkingDirOnCompletion()) {
            getRemoteWorkingDirectory().deleteRecursively();
        }

        if (remoteConn != null)
            closeQuietly(remoteConn);

        remoteWorkingDir = null;
        localConn = null;
        remoteConn = null;
    }

    public String getRemoteWorkingDirPath() {
        return remoteWorkingDirPath;
    }

    public void setRemoteWorkingDirPath(String remoteWorkingDirPath) {
        this.remoteWorkingDirPath = remoteWorkingDirPath;
    }

    public boolean isRetainRemoteWorkingDirOnCompletion() {
        return retainRemoteWorkingDirOnCompletion;
    }

    public void setRetainRemoteWorkingDirOnCompletion(boolean deleteWorkingDirOnCompletion) {
        this.retainRemoteWorkingDirOnCompletion = deleteWorkingDirOnCompletion;
    }

    protected OverthereFile uploadToWorkingDirectory(String content, String fileName) {
        getCtx().logOutput("Uploading file " + fileName + " to working directory.");
        OverthereFile target = getRemoteWorkingDirectory().getFile(fileName);
        OverthereUtils.write(content.getBytes(), target);
        return target;
    }

    protected OverthereFile uploadToWorkingDirectory(File content, String fileName) {
        String fileType = content.isDirectory() ? "directory" : "file";
        getCtx().logOutput("Uploading " + fileType + " " + fileName + " to working directory.");
        OverthereFile target = getRemoteWorkingDirectory().getFile(fileName);
        LocalFile.valueOf(content).copyTo(target);
        return target;
    }

    protected OverthereFile uploadToWorkingDirectory(URL content, String fileName) {
        getCtx().logOutput("Uploading file " + fileName + " to working directory.");
        OverthereFile target = getRemoteWorkingDirectory().getFile(fileName);
        OutputStream out = target.getOutputStream();
        try {
            Resources.copy(content, out);
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        } finally {
            closeQuietly(out);
        }
        return target;
    }

    public boolean hostFileExists(String remoteFile) {
        checkNotNull(emptyToNull(remoteFile));
        OverthereFile file = getRemoteConnection().getFile(remoteFile);
        return file.exists();
    }

    public String getHostLineSeparator() {
        return getContainer().getHost().getOs().getLineSeparator();
    }

    public String getHostFileSeparator() {
        return getContainer().getHost().getOs().getFileSeparator();
    }

    public String readHostFile(final String remoteFile) {
        checkNotNull(emptyToNull(remoteFile));
        OverthereFile file = getRemoteConnection().getFile(remoteFile);
        checkArgument(file.exists(), "File %s does not exist on host %s", remoteFile, getContainer().getHost());
        try {
            InputStream in = file.getInputStream();
            try {
                byte[] bytes = ByteStreams.toByteArray(in);
                return new String(bytes);
            } finally {
                closeQuietly(in);
            }
        } catch (IOException e) {
            throw new RuntimeIOException("Failed to read file " + remoteFile, e);
        }
    }

    public String[] readHostFileLines(final String remoteFile) {
        String data = readHostFile(remoteFile);
        Iterable<String> iterable = Splitter.on(getHostLineSeparator()).split(data);
        return Iterables.toArray(iterable, String.class);
    }

    public void createOrReplaceHostFile(String content, String remoteFile) {
        checkNotNull(emptyToNull(remoteFile));
        checkNotNull(content);
        OverthereFile file = getRemoteConnection().getFile(remoteFile);
        if (file.exists()) {
            logger.debug("File " + remoteFile + " already exists. Will delete before attempting to write.");
            file.delete();
        }

        final ByteArrayInputStream from = new ByteArrayInputStream(content.getBytes());
        final OutputStream to = file.getOutputStream();
        try {
            ByteStreams.copy(from, to);
        } catch (IOException e) {
            throw new RuntimeIOException("Failed to write to " + remoteFile, e);
        } finally {
            closeQuietly(from);
            closeQuietly(to);
        }
    }

    public NodeModel readHostXmlFileAsModel(final String remoteXmlFile) {
        checkNotNull(emptyToNull(remoteXmlFile));
        OverthereFile file = getRemoteConnection().getFile(remoteXmlFile);
        checkArgument(file.exists(), "File %s does not exist on host %s", remoteXmlFile, getContainer().getHost());
        InputStream remoteXmlStream = file.getInputStream();
        try {
            return NodeModel.parse(new InputSource(remoteXmlStream), false, true);
        } catch (ParserConfigurationException exc) {
            throw new RuntimeException("Cannot read xml file " + remoteXmlFile, exc);
        } catch (SAXException exc) {
            throw new RuntimeException("Cannot read xml file " + remoteXmlFile, exc);
        } catch (IOException exc) {
            throw new RuntimeIOException("Cannot read xml file " + remoteXmlFile, exc);
        } finally {
            closeQuietly(remoteXmlStream);
        }
    }

    /**
     * @deprecated Use {@link #evaluateTemplate(String, Map, boolean)} instead.
     */
    public String evaluateTemplate(String templatePath, Map<String, Object> vars) {
        return evaluateTemplate(templatePath, vars, false);
    }

    public String evaluateTemplate(String templatePath, Map<String, Object> vars, boolean maskPasswords) {
        Configuration cfg = ConfigurationHolder.getConfiguration();
        try {
            Template template = cfg.getTemplate(templatePath);
            StringWriter sw = new StringWriter();
            template.createProcessingEnvironment(vars, sw, new CiAwareObjectWrapper(new WorkingFolderUploader(getCtx()), maskPasswords)).process();
            return sw.toString();
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        } catch (TemplateException e) {
            throw new RuntimeException(e);
        }
    }

    protected void evaluateTemplate(OverthereFile renderTo, String templatePath, Map<String, Object> vars) {
        Configuration cfg = ConfigurationHolder.getConfiguration();
        OutputStream out = renderTo.getOutputStream();
        try {
            Template template = cfg.getTemplate(templatePath);
            template.process(vars, new OutputStreamWriter(out));
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        } catch (TemplateException e) {
            throw new RuntimeException(e);
        } finally {
            closeQuietly(out);
        }
    }

    public String resolveOsSpecificTemplate(String template) {
        if (classpathResourceExists(template + FREEMARKER_FILE_EXT)) {
            return template + FREEMARKER_FILE_EXT;
        }

        String osSpecificTemplate = template;

        String scriptExt = substringAfterLast(osSpecificTemplate, '.');
        if (scriptExt == null) {
            OperatingSystemFamily os = getContainer().getHost().getOs();
            osSpecificTemplate = osSpecificTemplate + os.getScriptExtension();
        }

        if (!classpathResourceExists(osSpecificTemplate)) {
            String osSpecificScriptTemplate = osSpecificTemplate + FREEMARKER_FILE_EXT;
            if (!classpathResourceExists(osSpecificScriptTemplate))
                throw new IllegalArgumentException("Resource " + osSpecificTemplate + " not found in classpath");
            else
                osSpecificTemplate = osSpecificScriptTemplate;
        }

        return osSpecificTemplate;
    }

    public boolean classpathResourceExists(String resource) {
        return Thread.currentThread().getContextClassLoader().getResource(resource) != null;
    }

    public static  String substringAfterLast(String str, char sub, String defaultValue) {
        String s = substringAfterLast(str, sub);
        if (s == null) {
            return defaultValue;
        }
        return s;
    }

    public static String substringAfterLast(String str, char sub) {
        int pos = str.lastIndexOf(sub);
        if (pos == -1) {
            return null;
        } else {
            return str.substring(pos + 1);
        }
    }

    private class WorkingFolderUploader implements FileUploader {
        private Map<String, String> uploadedFiles = newHashMap();
        private ExecutionContext ctx;

        public WorkingFolderUploader(ExecutionContext ctx) {
            this.ctx = ctx;
        }

        @Override
        public String upload(OverthereFile file) {
            if (uploadedFiles.containsKey(file.getName())) {
                return uploadedFiles.get(file.getName());
            }
            OverthereFile uploadedFile = getRemoteWorkingDirectory().getFile(file.getName());
            if (!uploadedFile.exists()) {
                String fileType = file.isDirectory() ? "directory" : "file";
                ctx.logOutput("Uploading " + fileType + " " + file.getName() + " to working directory.");
                file.copyTo(uploadedFile);
            } else {
                logger.warn("Not uploading file " + file.getName() + " to working directory again because it has already been uploaded.");
            }
            uploadedFiles.put(file.getName(), uploadedFile.getPath());
            return uploadedFile.getPath();
        }
    }

    protected final Logger logger = LoggerFactory.getLogger(this.getClass());

}
