package com.xebialabs.deployit.plugin.generic.step;

import java.io.File;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

import com.google.common.base.Preconditions;
import com.google.common.io.Resources;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Preview;
import com.xebialabs.deployit.plugin.api.flow.PreviewStep;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.inspection.InspectionContext;
import com.xebialabs.deployit.plugin.api.services.Repository;
import com.xebialabs.deployit.plugin.overthere.DefaultExecutionOutputHandler;
import com.xebialabs.deployit.plugin.overthere.HostContainer;
import com.xebialabs.overthere.CmdLine;
import com.xebialabs.overthere.OperatingSystemFamily;
import com.xebialabs.overthere.OverthereFile;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.io.Closeables.closeQuietly;
import static com.xebialabs.deployit.plugin.overthere.DefaultExecutionOutputHandler.handleStderr;
import static com.xebialabs.deployit.plugin.overthere.DefaultExecutionOutputHandler.handleStdout;
import static com.xebialabs.deployit.plugin.remoting.preview.PreviewOverthereConnection.getPreviewConnection;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.dumpScript;

@SuppressWarnings("serial")
public abstract class BaseExecutionStep extends BaseStep implements PreviewStep {

    private static final String MDC_KEY_SCRIPT_PATH = "scriptPath";

    private String scriptTemplatePath;
    private Map<String, Object> vars;

    private File artifact;

    private List<File> fileResources = newArrayList();
    private List<String> classpathResources = newArrayList();
    private List<String> templateClasspathResources = newArrayList();
    
    private Preview preview;

    public BaseExecutionStep(String scriptPath, HostContainer container, Map<String, Object> vars, String description) {
        super(description, container);
        this.scriptTemplatePath = scriptPath;
        this.vars = newHashMap(vars);
        Preconditions.checkNotNull(scriptTemplatePath);
    }

    @Override
    public StepExitCode doExecute() throws Exception {
        MDC.put(MDC_KEY_SCRIPT_PATH, scriptTemplatePath);
        try {
            uploadArtifactIfPresent();
            uploadFileResources();
            uploadClasspathResources();
            uploadTemplateClasspathResources();
            OverthereFile executable = uploadScriptToExecute(scriptTemplatePath);
            setWorkingDirectory();
            return executeScript(executable);
        } finally {
            MDC.remove(MDC_KEY_SCRIPT_PATH);
        }
    }

    @Override
    public Preview getPreview() {
        if (preview == null) {
            ctx = new ExecutionContext() {
                @Override
                public void logOutput(String output) {
                }

                @Override
                public void logError(String error, Throwable t) {
                }

                @Override
                public void logError(String error) {
                }

                @Override
                public Object getAttribute(String name) {
                    throw new UnsupportedOperationException();
                }

                @Override
                public void setAttribute(String name, Object value) {
                }

                @Override
                public Repository getRepository() {
                    throw new UnsupportedOperationException();
                }

                @Override
                public InspectionContext getInspectionContext() {
                    throw new UnsupportedOperationException();
                }

            };
            getLocalConnection();
            remoteConn = getPreviewConnection();
            try {
                String sourcePath = resolveOsSpecificTemplate(scriptTemplatePath);
                String contents = evaluateTemplate(sourcePath, vars, true);
                preview = Preview.withSourcePathAndContents(sourcePath, contents);
            } finally {
                ctx = null;
                disconnect();
            }
        }
        return preview;
    }

    protected OverthereFile uploadArtifactIfPresent() {
        if (artifact != null) {
            return uploadToWorkingDirectory(artifact, artifact.getName());
        }
        return null;
    }

    protected void uploadFileResources() {
        for (File fileResource : fileResources) {
            uploadToWorkingDirectory(fileResource, fileResource.getName());
        }
    }

    protected void uploadClasspathResources() {
        for (String cpResource : classpathResources) {
            String fileName = substringAfterLast(cpResource, '/', cpResource);
            uploadToWorkingDirectory(Resources.getResource(cpResource), fileName);
        }
    }

    protected void uploadTemplateClasspathResources() {
        for (String cpTemplate : templateClasspathResources) {
            uploadTemplateResourceToWorkingDirectory(cpTemplate);
        }
    }

    private OverthereFile uploadTemplateResourceToWorkingDirectory(String template) {
        String osSpecificTemplate = resolveOsSpecificTemplate(template);
        String content = evaluateTemplate(osSpecificTemplate, vars, false);

        String fileName = resolveOsSpecificFileName(osSpecificTemplate);
        return uploadToWorkingDirectory(content, fileName);
    }

    protected OverthereFile uploadScriptToExecute(String scriptTemplatePath) {
        String osSpecificTemplate = resolveOsSpecificTemplate(scriptTemplatePath);
        String generatedScript = evaluateTemplate(osSpecificTemplate, vars, false);

        dumpScript(scriptTemplatePath, generatedScript, scriptsLogger);

        String fileName = resolveOsSpecificFileName(osSpecificTemplate);
        return uploadToWorkingDirectory(generatedScript, fileName);
    }

    protected void setWorkingDirectory() {
        getRemoteConnection().setWorkingDirectory(getRemoteWorkingDirectory());
    }

    protected StepExitCode executeScript(OverthereFile executable) {
        executable.setExecutable(true);

        CmdLine cmdLine = CmdLine.build(executable.getPath());
        DefaultExecutionOutputHandler stdoutHandler = handleStdout(getCtx());
        DefaultExecutionOutputHandler stderrHandler = handleStderr(getCtx());
        try {
            getCtx().logOutput("Executing " + executable.getPath() + " on host " + getContainer().getHost());
            int rc = getRemoteConnection().execute(stdoutHandler, stderrHandler, cmdLine);
            if (rc != 0) {
                getCtx().logError("Execution failed with return code " + rc);
                return StepExitCode.FAIL;
            }
            return StepExitCode.SUCCESS;
        } finally {
            closeQuietly(stdoutHandler);
            closeQuietly(stderrHandler);
        }
    }

    public String getScriptPath() {
        return scriptTemplatePath;
    }

    public List<File> getFileResources() {
        return fileResources;
    }

    public void setFileResources(List<File> fileResources) {
        this.fileResources = fileResources;
    }

    public List<String> getClasspathResources() {
        return classpathResources;
    }

    public void setClasspathResources(List<String> classpathResources) {
        this.classpathResources = classpathResources;
    }

    public List<String> getTemplateClasspathResources() {
        return templateClasspathResources;
    }

    public void setTemplateClasspathResources(List<String> templateClasspathResources) {
        this.templateClasspathResources = templateClasspathResources;
    }

    public File getArtifact() {
        return artifact;
    }

    public void setArtifact(File artifact) {
        this.artifact = artifact;
    }

    public String getUploadedArtifactPath() {
        if (artifact != null) {
            return getRemoteWorkingDirectory().getPath() + getContainer().getHost().getOs().getFileSeparator() + artifact.getName();
        }
        return null;
    }

    protected String resolveOsSpecificFileName(String template) {
           return resolveOsSpecificFileName(template, getContainer());
    }

    public static String resolveOsSpecificFileName(String template, HostContainer hostContainer) {
         String osSpecificFile = substringAfterLast(template, '/', template);
        if (osSpecificFile.endsWith(FREEMARKER_FILE_EXT))
            osSpecificFile = osSpecificFile.substring(0, osSpecificFile.lastIndexOf(FREEMARKER_FILE_EXT));

        String fileExt = substringAfterLast(osSpecificFile, '.');
        if (fileExt == null) {
            OperatingSystemFamily os = hostContainer.getHost().getOs();
            osSpecificFile = osSpecificFile + os.getScriptExtension();
        }

        return osSpecificFile;
    }

    private static final Logger scriptsLogger = LoggerFactory.getLogger("com.xebialabs.deployit.plugin.generic.scripts");

}
