package com.xebialabs.deployit.plugin.generic.step;

import com.google.common.base.Preconditions;

import com.xebialabs.deployit.plugin.api.flow.StageableStep;
import com.xebialabs.deployit.plugin.api.flow.StagedFile;
import com.xebialabs.deployit.plugin.api.flow.StagingContext;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.udm.DeployableArtifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.generic.deployed.CopiedArtifact;
import com.xebialabs.deployit.plugin.overthere.HostContainer;
import com.xebialabs.overthere.OverthereFile;

import static com.google.common.base.Preconditions.checkNotNull;

@SuppressWarnings("serial")
public class ArtifactCopyStep extends BaseDeploymentStep implements StageableStep {

    private String targetPath;
    private boolean createTargetPath;
    private String targetFileName;
    private OverthereFile sourceFile;
    private String sourceFileDescription;
    private String description;
    private boolean preserveExistingFiles;
    private Artifact artifact;
    private StagedFile stagedArtifact;

    public ArtifactCopyStep(int order, OverthereFile sourceFile, HostContainer container, String targetPath) {
        super(order, container);
        this.sourceFile = sourceFile;
        this.targetPath = targetPath;
        Preconditions.checkNotNull(targetPath);
    }

    public ArtifactCopyStep(CopiedArtifact<? extends DeployableArtifact> copiedArtifact) {
        this(copiedArtifact.getCreateOrder(), copiedArtifact.getFile(), copiedArtifact.getContainer(), copiedArtifact.getTargetDirectory());
        artifact = copiedArtifact;
    }

    @Override
    public void requestStaging(final StagingContext ctx) {
        if (artifact != null) {
            logger.debug("[{}] has requesting staging of [{}]", this, artifact);
            stagedArtifact = ctx.stageArtifact(artifact, container.getHost());
        } else {
            logger.debug("[{}] has not requesting artifact staging.", this);
        }
    }

    @Override
    protected StepExitCode doExecute() throws Exception {
        if(stagedArtifact != null){
            return doCopy(stagedArtifact.get(getRemoteConnection(), getCtx()));
        }else{
            return doCopy(resolveSourceFile());
        }
    }

    private StepExitCode doCopy(OverthereFile sourceFile) throws Exception{
        Preconditions.checkNotNull(targetFileName);

        OverthereFile remoteDir = getRemoteConnection().getFile(targetPath);

        if (!remoteDir.exists()) {
            if (createTargetPath) {
                getCtx().logOutput("Creating path " + targetPath + " on host " + getContainer().getHost());
                remoteDir.mkdirs();
            } else if (sourceFile.isFile()) {
                getCtx().logError("Path " + targetPath + " on host " + getContainer().getHost() + " does not exist.");
                return StepExitCode.FAIL;
            }
        } else {
            if (preserveExistingFiles) {
                getCtx().logOutput(remoteDir.getPath() + " already exists on host " + getContainer().getHost() + ". Leave it.");
                return StepExitCode.SUCCESS;
            }
        }

        OverthereFile remoteFile;
        if (sourceFile.isDirectory()) {
            remoteFile = remoteDir;
        } else {
            remoteFile = remoteDir.getFile(getTargetFileName());
            if (remoteFile.exists() && !preserveExistingFiles) {
                getCtx().logOutput(remoteFile.getPath() + " already exists on host " + getContainer().getHost() + ". Will replace.");
                remoteFile.delete();
            }
        }

        getCtx().logOutput("Copying " + sourceFile.getPath() + " to " + remoteFile.getPath() + " on host " + getContainer().getHost());
        sourceFile.copyTo(remoteFile);
        return StepExitCode.SUCCESS;
    }

    protected OverthereFile resolveSourceFile() {
        checkNotNull(sourceFile);
        return sourceFile;
    }

    @Override
    public String getDescription() {
        if (description == null) {
            return "Copy " + getSourceFileDescription() + " to " + getContainer().getHost();
        }
        return description;
    }

    public void setDescription(String description) {
        this.description = description;
    }

    public String getSourceFileDescription() {
        return sourceFileDescription;
    }

    public void setSourceFileDescription(String sourceFileDescription) {
        this.sourceFileDescription = sourceFileDescription;
    }

    public boolean isCreateTargetPath() {
        return createTargetPath;
    }

    public void setCreateTargetPath(boolean createTargetPath) {
        this.createTargetPath = createTargetPath;
    }

    public String getTargetFileName() {
        return targetFileName;
    }

    public void setTargetFileName(String targetFileName) {
        this.targetFileName = targetFileName;
    }

    public boolean isPreserveExistingFiles() {
        return preserveExistingFiles;
    }

    public void setPreserveExistingFiles(final boolean preserveExistingFiles) {
        this.preserveExistingFiles = preserveExistingFiles;
    }
}
