package com.xebialabs.deployit.plugin.generic.container;

import static com.google.common.collect.Maps.newHashMap;

import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.Sets;

import com.xebialabs.deployit.plugin.api.deployment.planning.Contributor;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.generic.ci.BaseGenericContainer;
import com.xebialabs.deployit.plugin.generic.ci.Container;
import com.xebialabs.deployit.plugin.generic.ci.NestedContainer;
import com.xebialabs.deployit.plugin.generic.deployed.AbstractDeployed;
import com.xebialabs.deployit.plugin.generic.step.ScriptExecutionStep;
import com.xebialabs.deployit.plugin.generic.step.WaitStep;
import com.xebialabs.deployit.plugin.overthere.Host;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Strings.nullToEmpty;
import static com.google.common.collect.Lists.newArrayList;
import static java.lang.String.format;

public class LifeCycleContributor {

    @Contributor
    public void restartContainers(Deltas deltas, DeploymentPlanningContext plan) {
        Set<BaseGenericContainer> containers = gatherTargets(deltas.getDeltas());
        for (BaseGenericContainer container : containers) {
            if (!isNullOrEmpty(container.getRestartScript())) {
                addRestartSteps(plan, container);
            } else {
                addStartAndStopSteps(plan, container);
            }
        }
    }

    private static void addStartAndStopSteps(DeploymentPlanningContext plan, BaseGenericContainer target) {
        checkArgument(!isNullOrEmpty(target.getStartScript()), format("%s start script must be specified when no restart script defined.", target.getId()));
        checkArgument(!isNullOrEmpty(target.getStopScript()), format("%s stop script must be specified when no restart script defined.", target.getId()));

        plan.addStep(createScriptStep("Start", target, target.getStartScript(), target.getStartOrder(), target.getStartClasspathResources(), target.getStartTemplateClasspathResources()));
        addWaitStep(plan, "start", target, target.getStartWaitTime(), target.getStartOrder());

        plan.addStep(createScriptStep("Stop", target, target.getStopScript(), target.getStopOrder(), target.getStopClasspathResources(),target.getStopTemplateClasspathResources()));
        addWaitStep(plan, "stop", target, target.getStopWaitTime(), target.getStopOrder());
    }

    private static void addRestartSteps(DeploymentPlanningContext plan, BaseGenericContainer target) {
        Step step = createScriptStep("Restart", target, target.getRestartScript(), target.getRestartOrder(), target.getRestartClasspathResources(), target.getRestartTemplateClasspathResources());
        plan.addStep(step);
        addWaitStep(plan, "restart", target, target.getRestartWaitTime(), target.getRestartOrder());
    }

    private static void addWaitStep(DeploymentPlanningContext plan, String action, BaseGenericContainer target, int waitTime, int scriptOrder) {
        if (waitTime > 0) {
            plan.addStep(new WaitStep(scriptOrder + 1, waitTime, target.getName(), action));
        }
    }

    private static Step createScriptStep(String verb, BaseGenericContainer container, String script, int scriptOrder, Set<String> classpathResources, Set<String> templateClasspathResources) {
        Map<String, Object> vars = newHashMap();
        vars.put("container", container);
        Host lifecycleConnectionHost = container.getStopStartRestartConnection();
        ScriptExecutionStep step;
        if (lifecycleConnectionHost != null) {
            String description = format("%s using connection settings for %s", getDescription(verb, container), lifecycleConnectionHost.getName());
            step = new ScriptExecutionStep(scriptOrder, script, lifecycleConnectionHost, vars, description);
        } else {
            step = new ScriptExecutionStep(scriptOrder, script, container, vars, getDescription(verb, container));
        }
        step.setClasspathResources(newArrayList(classpathResources));
        step.setTemplateClasspathResources(newArrayList(templateClasspathResources));
        return step;
    }



    private static String getDescription(String verb, BaseGenericContainer container) {
        return format("%s %s", verb, container.getName());
    }


    private static Set<BaseGenericContainer> gatherTargets(List<Delta> operations) {
        final Set<BaseGenericContainer> targets = Sets.newTreeSet();
        for (Delta operation : operations) {
            addTarget(targets, operation.getOperation(),  operation.getDeployed());
            addTarget(targets, operation.getOperation(), operation.getPrevious());
        }
        return targets;
    }

    private static void addTarget(Set<BaseGenericContainer> targets, Operation operation, Deployed<?, ?> deployed) {
        if (deployed == null) {
            return;
        }

        if (deployed instanceof AbstractDeployed) {
            AbstractDeployed<?> abstractDeployed = (AbstractDeployed<?>) deployed;
            boolean restartRequired = abstractDeployed.isRestartRequired();
            if (restartRequired) {
                if (operation != Operation.NOOP || abstractDeployed.isRestartRequiredForNoop()) {
                    targets.add(getContainerToRestart(deployed.getContainer()));
                }
            }
        }
    }

    private static BaseGenericContainer getContainerToRestart(com.xebialabs.deployit.plugin.api.udm.Container c) {
        if (c instanceof Container) {
            return (Container) c;
        } else if (c instanceof NestedContainer) {
            NestedContainer nc = (NestedContainer) c;
            if(!isNullOrEmpty(nc.getStartScript()) || !isNullOrEmpty(nc.getStopScript()) || !isNullOrEmpty(nc.getRestartScript())) {
                return nc;
            } else {
                return getContainerToRestart(nc.getParentContainer());
            }
        } else {
            throw new IllegalStateException(format("Container [%s] is not a generic (nested) container", c));
        }
    }
    
    private static boolean isNullOrEmpty(String s) {
        return nullToEmpty(s).trim().isEmpty();
    }

}
