package com.xebialabs.deployit.plugin.generic.step;

import java.io.IOException;
import java.io.StringWriter;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.mail.MessagingException;
import org.slf4j.MDC;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Preview;
import com.xebialabs.deployit.plugin.api.flow.PreviewStep;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.rules.RulePostConstruct;
import com.xebialabs.deployit.plugin.api.rules.StepMetadata;
import com.xebialabs.deployit.plugin.api.rules.StepParameter;
import com.xebialabs.deployit.plugin.api.rules.StepPostConstructContext;
import com.xebialabs.deployit.plugin.generic.freemarker.ConfigurationHolder;
import com.xebialabs.deployit.plugin.mail.SmtpServer;
import com.xebialabs.deployit.plugin.overthere.HostContainer;
import com.xebialabs.deployit.plugin.steps.ContextHelper;

import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateException;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;

@SuppressWarnings("serial")
@StepMetadata(name = "manual")
public class InstructionStep implements Step, PreviewStep {

    private static final String MDC_KEY_TEMPLATE_PATH = "templatePath";

    @StepParameter(description = "The execution order of the step")
    private Integer order;

    @StepParameter(description = "Description of this step, as it should appear in generated deployment plans")
    private String description;

    @StepParameter(name = "freemarkerContext", description = "Dictionary that contains all values available in the template", required = false, calculated = true)
    private Map<String, Object> vars = new HashMap<>();

    @StepParameter(name = "messageTemplate", description = "The path to the email template to display and send out")
    private String templatePath;

    @StepParameter(name = "mailTo", description = "The list of email receivers to send instructions to")
    private List<String> toAddresses;

    @StepParameter(description = "The email subject line", calculated = true)
    private String subject;

    @StepParameter(name = "mailFrom", description = "The email's sender ('From:') email address")
    private String fromAddress;

    @StepParameter(description = "Mail server that is used to send emails", calculated = true)
    private SmtpServer mailServer;

    private boolean paused = false;

    private Preview preview;

    @SuppressWarnings("UnusedDeclaration")
    public InstructionStep() {
    }

    public InstructionStep(int order, String description, HostContainer container, Map<String, Object> vars, String templatePath) {
        this.order = order;
        this.description = description;
        this.vars = newHashMap(vars);
        this.vars.put("step", this);
        this.templatePath = templatePath;
        this.toAddresses = newArrayList();
        Preconditions.checkNotNull(templatePath);
    }

    @RulePostConstruct
    protected void doPostConstruct(final StepPostConstructContext ctx) {
        if (mailServer == null) {
            mailServer = SmtpServer.getMailServer(ctx.getDeployedApplication().getEnvironment(), ctx.getRepository());
        }
        vars = ContextHelper.defaultContext(ctx, vars);
        if (subject == null) subject = description;
    }

    @Override
    public int getOrder() {
        return order;
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public StepExitCode execute(ExecutionContext ctx) throws Exception {
        MDC.put(MDC_KEY_TEMPLATE_PATH, templatePath);
        try {
            String instructions = renderEmailTemplate(templatePath, vars);
            if (!paused) {
                sendInstructions(ctx, instructions);
                return pauseExecution();
            } else {
                return continueExecution(ctx, instructions);
            }
        } finally {
            MDC.remove(MDC_KEY_TEMPLATE_PATH);
        }
    }

    private void sendInstructions(final ExecutionContext ctx, final String instructions) {
        mailInstructions(instructions, ctx);
        ctx.logOutput(instructions);
    }

    private StepExitCode pauseExecution() {
        paused = true;
        return StepExitCode.PAUSE;
    }

    private StepExitCode continueExecution(final ExecutionContext ctx, final String instructions) {
        ctx.logOutput("Assuming manual process performed. Continuing...");
        ctx.logOutput("------------");
        ctx.logOutput("Instructions");
        ctx.logOutput("------------");
        ctx.logOutput(instructions);
        return StepExitCode.SUCCESS;
    }

    protected void mailInstructions(String instructions, ExecutionContext ctx) {
        if (!toAddresses.isEmpty() && mailServer != null) {
            ctx.logOutput("Mailing instructions to " + Joiner.on(',').join(toAddresses));
            subject = (Strings.isNullOrEmpty(subject)) ? getDescription() : subject;
            try {
                mailServer.sendMessage(subject, instructions, toAddresses, fromAddress);
            } catch (MessagingException e) {
                ctx.logError("Failed to send mail.", e);
                ctx.logOutput(Strings.repeat("-", 50));
            }
        } else {
            ctx.logOutput("Not sending email instructions.");
        }
    }

    protected String renderEmailTemplate(String template, Map<String, Object> vars) throws IOException, TemplateException {
        Configuration cfg = ConfigurationHolder.getConfiguration();
        Template loadedTemplate = cfg.getTemplate(template);
        StringWriter sw = new StringWriter();
        loadedTemplate.process(vars, sw);
        return sw.toString();
    }

    public List<String> getToAddresses() {
        return toAddresses;
    }

    public void setToAddresses(List<String> toAddresses) {
        this.toAddresses = toAddresses;
    }

    public String getSubject() {
        return subject;
    }

    public void setSubject(String subject) {
        this.subject = subject;
    }

    public String getFromAddress() {
        return fromAddress;
    }

    public void setFromAddress(String fromAddress) {
        this.fromAddress = fromAddress;
    }

    public SmtpServer getMailServer() {
        return mailServer;
    }

    public void setMailServer(SmtpServer mailServer) {
        this.mailServer = mailServer;
    }


    @Override
    public Preview getPreview() {
        if (preview == null) {
            try {
                preview = Preview.withSourcePathAndContents(templatePath, renderEmailTemplate(templatePath, vars));
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        }
        return preview;
    }
}

