# Get provided website name or use the deployed name.
$websiteName = if($deployed.websiteName) { $deployed.websiteName } else { $deployed.name }

# Verify website name is not being changed.
if($previousDeployed) {
    $previousWebsiteName = if($previousDeployed.websiteName) { $previousDeployed.websiteName } else { $previousDeployed.name }
    if($websiteName -ne $previousWebsiteName) {
        Write-Output "Renaming a website is not supported. Undeploy and deploy the website instead."
        Exit 1
    }
}

# Create physical path if it doesn't exist yet.
if (!(Test-Path $deployed.physicalPath)) {
    Write-Output "Creating empty website physical path [$($deployed.physicalPath)]."
    if(!(New-Item $deployed.physicalPath -Type Directory -Force)) {
        Write-Output "Cannot create directory [$($deployed.physicalPath)]."
        Exit 1
    }
}

# Check whether website already exists and then either modify or create it.
$websitePath = "IIS:\Sites\$websiteName"
if (Test-Path $websitePath) {
    Write-Output "Modifying existing website [$websiteName]."
 
    Set-ItemProperty -Path $websitePath -Name PhysicalPath -Value $deployed.physicalPath
} else {
    Write-Output "Creating new website [$websiteName]."

    $params = @{ Name=$websiteName; PhysicalPath=$deployed.physicalPath }

    if (!(Get-Website)) {
        Write-Output "INFO: Activating workaround for PowerShell bug that occurs when there are no websites yet."
        $params.id = 1
    }

    New-Website @params | Out-Null
}

# Set application pool.
if($deployed.applicationPoolName) {
    Set-ItemProperty -Path $websitePath -Name ApplicationPool -Value $deployed.applicationPoolName
}

# Remove existing bindings.
foreach ($existingBinding in (Get-WebBinding -Name $websiteName)) {
    Write-Output "Removing existing binding [$($existingBinding.bindingInformation)] from website [$websiteName]."
    Remove-WebBinding -InputObject $existingBinding
}

# Add bindings.
foreach ($binding in $deployed.bindings) {
    Write-Output "Adding binding [$($binding.name)] for website [$websiteName]."

    $params = @{ Name=$websiteName; Protocol=$binding.protocol; Port=$binding.port; IPAddress=$binding.ipAddress }

    if ($binding.hostHeader) {
        $params.hostHeader = $binding.hostHeader
    }

    New-WebBinding @params | Out-Null
}


