package com.xebialabs.deployit.plugin.iis;

import java.util.Calendar;
import java.util.List;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.nodetype.NodeType;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.server.api.repository.RawRepository;
import com.xebialabs.deployit.server.api.upgrade.Upgrade;
import com.xebialabs.deployit.server.api.upgrade.UpgradeException;
import com.xebialabs.deployit.server.api.upgrade.Version;

/**
 * Upgrades the Deployit repository to the format used by IIS plugin 3.9.0 and upx.
 */
public class UpgradeToIisPlugin390 extends Upgrade {

    @Override
    public Version upgradeVersion() {
        return Version.valueOf("iis-plugin", "3.9.0");
    }

    @Override
    public boolean doUpgrade(RawRepository repository) throws UpgradeException {
        try {
            createWebsiteBindingSpecs(repository);
            createWebsiteBindings(repository);
            return true;
        } catch (RepositoryException exc) {
            throw new UpgradeException("Cannot upgrade repository to format for " + upgradeVersion(), exc);
        }
    }

    private void createWebsiteBindingSpecs(RawRepository repository) throws RepositoryException {
        List<Node> websiteSpecs = repository.findNodesByType(Type.valueOf("iis.WebsiteSpec"));
        for (Node websiteSpec : websiteSpecs) {
            createWebsiteBindingSpec(repository, websiteSpec);
        }
    }

    private void createWebsiteBindingSpec(RawRepository repository, Node websiteSpec) throws RepositoryException {
        String port = "";
        if (websiteSpec.hasProperty("port")) {
            port = websiteSpec.getProperty("port").getString();
        }
        
        logger.info("Creating [iis.WebsiteBindingSpec] at [{}/binding1]", websiteSpec.getPath().substring(1));
        Node websiteBindingSpec = websiteSpec.addNode("binding1");
        websiteBindingSpec.addMixin("deployit:configurationItem");
        websiteBindingSpec.addMixin(NodeType.MIX_VERSIONABLE);
        websiteBindingSpec.setProperty("$configuration.item.type", "iis.WebsiteBindingSpec");
        websiteBindingSpec.setProperty("$lastModified", Calendar.getInstance());
        websiteBindingSpec.setProperty("port", port);
    }

    private void createWebsiteBindings(RawRepository repository) throws RepositoryException {
        List<Node> websites = repository.findNodesByType(Type.valueOf("iis.Website"));
        for (Node website : websites) {
            createWebsiteBinding(repository, website);
        }
    }

    private void createWebsiteBinding(RawRepository repository, Node website) throws RepositoryException {
        int port = 80;
        if (website.hasProperty("port")) {
            port = (int) website.getProperty("port").getLong();
        }
        Node websiteBinding = website.addNode("binding1");
        
        logger.info("Creating [iis.WebsiteBinding] at [{}/binding1]", website.getPath().substring(1));
        websiteBinding.addMixin("deployit:configurationItem");
        websiteBinding.addMixin(NodeType.MIX_VERSIONABLE);
        websiteBinding.setProperty("$configuration.item.type", "iis.WebsiteBinding");
        websiteBinding.setProperty("$lastModified", Calendar.getInstance());
        websiteBinding.setProperty("deployable", website.getProperty("deployable").getNode().getNode("binding1"));
        websiteBinding.setProperty("port", port);
    }

    private static Logger logger = LoggerFactory.getLogger(UpgradeToIisPlugin390.class);

}
