#
######################################################################################################################
# common functions
######################################################################################################################
$mappings = @{
    "logonMethod" = @{0 = "Interactive"; 1 = "Batch"; 2 = "Network"; 3 = "ClearText"};
    "cookieless" = @{0 = "UseUri"; 1 = "UseCookies"; 2 = "AutoDetect"; 3 = "UseDeviceProfile"};
    "protection" = @{0 = "All"; 1 = "None"; 2 = "Encryption"; 3 = "Validation"};
}

Function parseValue($name, $value) {
    if(("$value" -eq "False") -or ("$value" -eq "True")) {
        return ([string]$value).ToLower()
    } elseif ($mappings.ContainsKey($name)) {
        return $mappings.Get_Item($name).Get_Item($value)
    } else {
        return $value
    }
}

Function convertToHash($object, $exclude=@()) {
    $hash = @{}
    foreach ($property in $object._properties) {
        if (!($exclude -contains $property)) {
            $value = $object.$property
            $hash[$property] = "$value"
        }
    }
    return $hash
}

Function inspectAttributes($id, $prefix, $object) {
    foreach ($attribute in $object.Attributes) {
        $name = $attribute.Name
        $value = parseValue $name $attribute.Value
        inspectedProperty $id "$prefix$name" $value
    }
}

######################################################################################################################
# authentication
######################################################################################################################

Function setAuthentication($deployed, $location) {
    foreach ($authentication in $deployed.authentication) {
        Write-Host "Applying authentication settings for [$location]."

        $path = "IIS:\Sites\$location".Replace("/", "\")
        $isVirtualDirectory = ("$($deployed.type)" -eq "iis.VirtualDirectory")

        # anonymousAuthentication
        if ("$($authentication.anonymousAuthentication.userIdentity)" -eq "ApplicationPool") {
            $authentication.anonymousAuthentication.userName = ""
            $authentication.anonymousAuthentication.password = ""
        }
        Set-WebConfiguration system.webServer/security/authentication/anonymousAuthentication -PSPath IIS:\ -Location $location -Value (convertToHash $authentication.anonymousAuthentication)

        # aspnetImpersonation
        Set-WebConfiguration system.web/identity $path -Value (convertToHash $authentication.aspnetImpersonation)

        # basicAuthentication
        Set-WebConfiguration system.webServer/security/authentication/basicAuthentication -PSPath IIS:\ -Location $location -Value (convertToHash $authentication.basicAuthentication)

        # clientCertificateMappingAuthentication
        Set-WebConfiguration system.webServer/security/authentication/clientCertificateMappingAuthentication -PSPath IIS:\ -Location $location -Value (convertToHash $authentication.clientCertificateMappingAuthentication)

        # digestAuthentication
        Set-WebConfiguration system.webServer/security/authentication/digestAuthentication -PSPath IIS:\ -Location $location -Value (convertToHash $authentication.digestAuthentication)

        # formsAuthentication
        if ($isVirtualDirectory) {
            Write-Host "Forms authentication options not supported on virtual directory elements. Ignoring..."
        } else {
            Set-WebConfiguration system.web/authentication/forms $path -Value (convertToHash $authentication.formsAuthentication)
        }

        # windowsAuthentication
        if ($authentication.windowsAuthentication.enabled) {
            Remove-WebConfigurationProperty -filter system.webServer/security/authentication/windowsAuthentication/providers -name "." -PSPath IIS:\ -Location $location
            foreach ($provider in $authentication.windowsAuthentication.providers) {
                Add-WebConfiguration -Filter system.webServer/security/authentication/windowsAuthentication/providers -Value $provider -PSPath IIS:\ -Location $location
            }
        }
        Set-WebConfiguration system.webServer/security/authentication/windowsAuthentication -PSPath IIS:\ -Location $location -Value (convertToHash $authentication.windowsAuthentication)

        if (!$isVirtualDirectory) {
            $config = Get-WebConfiguration system.web/authentication $path
            if ($authentication.anonymousAuthentication.enabled) {
                $config.mode = "None"
            }
            if ($authentication.formsAuthentication.enabled) {
                $config.mode = "Forms"
            }
            if ($authentication.windowsAuthentication.enabled) {
                $config.mode = "Windows"
            }
            $config | Set-WebConfiguration system.web/authentication
        }
    }
}

Function inspectAuthentication($id, $path) {
    $aid = "$id/Authentication"
    discovered $aid "iis.Authentication"

    $anonymousAuthentication = Get-WebConfiguration system.webServer/security/authentication/anonymousAuthentication $path
    inspectAttributes $aid "anonymousAuthentication_" $anonymousAuthentication
    if (!($anonymousAuthentication.userName)) {
        inspectedProperty $aid "anonymousAuthentication_userIdentity" "ApplicationPool"
    }
    inspectAttributes $aid "aspnetImpersonation_" (Get-WebConfiguration system.web/identity $path)
    inspectAttributes $aid "basicAuthentication_" (Get-WebConfiguration system.webServer/security/authentication/basicAuthentication $path)
    inspectAttributes $aid "clientCertificateMappingAuthentication_" (Get-WebConfiguration system.webServer/security/authentication/clientCertificateMappingAuthentication $path)
    inspectAttributes $aid "digestAuthentication_" (Get-WebConfiguration system.webServer/security/authentication/digestAuthentication $path)
    inspectAttributes $aid "formsAuthentication_" (Get-WebConfiguration system.web/authentication/forms $path)
    inspectAttributes $aid "windowsAuthentication_" (Get-WebConfiguration system.webServer/security/authentication/windowsAuthentication $path)

    $providers = Get-WebConfiguration system.webServer/security/authentication/windowsAuthentication/providers $path
    foreach ($provider in $providers.Collection) {
        inspectedProperty $aid "windowsAuthentication_providers" "$($provider.value)"
    }

    $authentication = Get-WebConfiguration system.web/authentication $path
    if ("$($authentication.mode)" -eq "Forms") {
        inspectedProperty $aid "formsAuthentication_enabled" "true"
    } elseif ("$($authentication.mode)" -eq "Windows") {
        inspectedProperty $aid "windowsAuthentication_enabled" "true"
    }
    inspectedItem $aid
}

######################################################################################################################
# directory browsing
######################################################################################################################

Function setDirectoryBrowsing($deployed, $location) {
    Write-Host "Applying directory browsing settings for [$location]."

    Set-WebConfiguration system.webServer/directoryBrowse -PSPath IIS:\ -Location $location -Value (convertToHash $deployed.directoryBrowse)
}

Function inspectDirectoryBrowsing($id, $path) {
	$directoryBrowse = Get-WebConfiguration system.webServer/directoryBrowse $path
    inspectedProperty $id "directoryBrowse_enabled" (parseValue "enabled" $directoryBrowse.enabled)
    $showFlags = $directoryBrowse.showFlags
    if(!$showFlags) {
        $showFlags = "None"
    }
    inspectedProperty $id "directoryBrowse_showFlags" $showFlags
}