package com.xebialabs.deployit.plugin.jbossdm.step;

import static com.xebialabs.deployit.plugin.remoting.vars.VarsConverter.STAGED_FILE_VARIABLE_NAME_PREFIX;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Preview;
import com.xebialabs.deployit.plugin.api.flow.PreviewStep;
import com.xebialabs.deployit.plugin.api.flow.StageableStep;
import com.xebialabs.deployit.plugin.api.flow.StagedFile;
import com.xebialabs.deployit.plugin.api.flow.StagingContext;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.rules.RulePostConstruct;
import com.xebialabs.deployit.plugin.api.rules.StepMetadata;
import com.xebialabs.deployit.plugin.api.rules.StepParameter;
import com.xebialabs.deployit.plugin.api.rules.StepPostConstructContext;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.jbossdm.container.CliManagedContainer;
import com.xebialabs.deployit.plugin.steps.CalculatedStep;
import com.xebialabs.deployit.plugin.steps.ContextHelper;
import com.xebialabs.deployit.plugin.steps.TargetContainerJavaHelper;

@StepMetadata(name = "jboss-cli")
public class JbossCliStep extends CalculatedStep implements PreviewStep, StageableStep {
    protected static final Logger logger = LoggerFactory.getLogger(JbossCliStep.class);
    public static final String AUTO_UPLOAD = "_AUTO_UPLOAD";


    @StepParameter(description = "Path to the python script to be executed.")
    private String script;

    @StepParameter(description = "Dictionary that represent context available to the python script", required = false, calculated = true)
    private Map<String, Object> pythonContext = new HashMap<>();

    @StepParameter(description = "CLI managed container (domain, standalone server, profile or managed group) where to execute the cli script",
        calculated = true)
    private CliManagedContainer container;

    @StepParameter(description = "List of python library scripts that should be automatically loaded when using a JBoss CLI script. Usage of additional script libraries is discouraged, please use proper Python modules.", required = false, calculated = true)
    private List<String> additionalLibraries = new ArrayList<>();

    @StepParameter(
        description = "If true, artifacts from the python context will be uploaded to the target host and available to the script as <bindingName>.file (of type OverthereFile). Defaults to true. If set to false it will prevent staging.")
    private Boolean uploadArtifactsInPythonContext = true;

    private BaseStep cliStep;

    @RulePostConstruct
    @Override
    public void doPostConstruct(StepPostConstructContext ctx) {
        super.doPostConstruct(ctx);
        pythonContext = ContextHelper.defaultContext(ctx, pythonContext);
        container = calculateTargetContainer(ctx);
        cliStep = new CliDeploymentStep(script, getOrder(), pythonContext, getDescription(), container);
        cliStep.setAdditionalLibraries(additionalLibraries);
    }

    private CliManagedContainer calculateTargetContainer(StepPostConstructContext ctx) {
        CliManagedContainer targetcontainer = container;
        if (null == targetcontainer) {
            targetcontainer = TargetContainerJavaHelper.defaultTargetContainer(ctx, CliManagedContainer.class);
        }
        return targetcontainer;
    }

    @Override
    public Preview getPreview() {
        return cliStep.getPreview();
    }

    @Override
    public void requestStaging(StagingContext ctx) {
        if (!uploadArtifactsInPythonContext) {
            return;
        }

        Map<String, Object> stagedEntries = new HashMap<String, Object>();
        for (Map.Entry<String, Object> entry : pythonContext.entrySet()) {
            Object entryVal = entry.getValue();
            if (entryVal instanceof Artifact) {
                Artifact artifact = (Artifact) entryVal;
                StagedFile stagedFile = ctx.stageArtifact(artifact, container.getHost());
                stagedEntries.put(entry.getKey() + STAGED_FILE_VARIABLE_NAME_PREFIX, stagedFile);
            }
        }
        pythonContext.putAll(stagedEntries);
    }

    @Override
    public StepExitCode execute(ExecutionContext ctx) throws Exception {
        StepExitCode exitCode = StepExitCode.FAIL;
        if (uploadArtifactsInPythonContext) {
            uploadArtifacts();
        }

        exitCode = cliStep.execute(ctx);

        return exitCode;
    }

    /**
     * Python scripts are executed locally, but interact with the CLI on remote system - users might expect
     * to have access to automatically uploaded artifacts on a target container from withing scripts.
     *
     * We scan variables of type Artifact in python context and mark local overthere file for upload (stage)
     * if it's not already staged.
     *
     * JBossCliStep will execute as python script and variables will be bound to the python script execution
     * context (in BaseStep). If variable is of type Artifact and it was staged (a key is present in context)
     * it will be wrapped by a dynamic proxy that will resolve file property to the remote overthere file.
     *
     * This means that 'deployed.file' will return OverthereFile on a remote container when referenced in
     * python scripts - which is suitable because file doesn't have to be explicitly uploaded to the working
     * directory.
     *
     * One can use uploadArtifactsInPythonContext boolean flag to turn off automatic upload and do revert
     * to manual upload.
     */
    private void uploadArtifacts() {
        TreeSet<String> sortedKeys = new TreeSet<>(pythonContext.keySet());
        for (String varName : sortedKeys) {
            Object varValue = pythonContext.get(varName);
            if (varValue instanceof Artifact) {
                markArtifactForUpload(varName, (Artifact) varValue);
            }
        }
    }

    private void markArtifactForUpload(String varName, Artifact artifact) {
        logger.debug("Converting variable [{}]", varName);
        if (!uploadArtifactsInPythonContext) {
            logger.debug("Property 'file' of [{}] will NOT be automatically uploaded and adjusted to point to a remote overthere file.", varName);
            return;
        }

        final String stagedFileVarName = varName + STAGED_FILE_VARIABLE_NAME_PREFIX;
        pythonContext.put(stagedFileVarName + AUTO_UPLOAD, Boolean.TRUE);
        if (pythonContext.containsKey(stagedFileVarName)) {
            return;
        }

        // mimic staging...
        JustInTimeFile file = new JustInTimeFile(artifact);
        pythonContext.put(stagedFileVarName, file);
    }

}
