package com.xebialabs.deployit.plugin.powershell;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.MDC_KEY_SCRIPT_PATH;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.dumpScript;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.loadScript;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.loadScriptDir;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.uploadScript;
import static com.xebialabs.overthere.util.OverthereUtils.getName;

import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

import com.google.common.io.Closeables;
import com.xebialabs.deployit.plugin.api.execution.ExecutionContext;
import com.xebialabs.deployit.plugin.api.execution.Step;
import com.xebialabs.deployit.plugin.overthere.ExecutionContextOverthereProcessOutputHandler;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.OverthereFile;

@SuppressWarnings("serial")
abstract class PowerShellStep<C extends ExecutionContext> implements Step<C> {

	private static final String STANDARD_RUNTIME_PATH = "powershell/runtime";
	private PowerShellContainer container;
	private String scriptPath;
	private Map<String, Object> powerShellVars;
	private String description;
    private boolean uploadArtifactData = true;

	public PowerShellStep(PowerShellContainer container, String scriptPath, Map<String, Object> powerShellVars, String description) {
		this.container = checkNotNull(container, "container is null");
		this.scriptPath = checkNotNull(scriptPath, "scriptPath is null");
		this.powerShellVars = checkNotNull(powerShellVars, "powerShellVars is null");
		this.description = checkNotNull(description, "description is null");
	}

	protected Result doExecute(ExecutionContext ctx) throws Exception {
		MDC.put(MDC_KEY_SCRIPT_PATH, scriptPath);
		try {
			OverthereConnection conn = container.getHost().getConnection();
			try {
				String finalScript = aggregateScript(conn);
				dumpScript(getName(scriptPath), finalScript, scriptsLogger);
				OverthereFile uploadedScriptFile = uploadScript(conn, scriptPath, finalScript);

				int res = executePowerShellScript(ctx, conn, uploadedScriptFile);

				return (res == 0) ? Result.Success : Result.Fail;
			} finally {
				Closeables.closeQuietly(conn);
			}
		} finally {
			MDC.remove(MDC_KEY_SCRIPT_PATH);
		}
	}

	protected String aggregateScript(OverthereConnection connection) {
		String convertedVars = PowerShellVarsConverter.javaToPowerShell(connection, powerShellVars, uploadArtifactData);

		StringBuilder b = new StringBuilder();
		b.append(loadScriptDir(STANDARD_RUNTIME_PATH));
		b.append(loadScriptDir(container.getRuntimePath()));
		b.append("# PowerShellVars\n");
		b.append(convertedVars);
		b.append(loadScript(scriptPath));
		
		return b.toString().replaceAll("\n", "\r\n");
	}

	private int executePowerShellScript(ExecutionContext ctx, OverthereConnection conn, OverthereFile script) {
		logger.info("Executing PowerShell script {} on {}", script, conn);
		return conn.execute(new ExecutionContextOverthereProcessOutputHandler(ctx), container.getScriptCommandLine(script));
	}

	@Override
	public String getDescription() {
		return description;
	}

    public void setUploadArtifactData(boolean uploadArtifactData) {
        this.uploadArtifactData = uploadArtifactData;
    }

	private static Logger logger = LoggerFactory.getLogger(PowerShellStep.class);

	private static final Logger scriptsLogger = LoggerFactory.getLogger("com.xebialabs.deployit.plugin.powershell.scripts");

}
