package com.xebialabs.deployit.plugin.powershell;

import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

import com.google.common.io.Closeables;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Preview;
import com.xebialabs.deployit.plugin.api.flow.PreviewStep;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.overthere.DefaultExecutionOutputHandler;
import com.xebialabs.deployit.plugin.overthere.HostContainer;
import com.xebialabs.overthere.CmdLine;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.OverthereFile;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.io.Closeables.closeQuietly;
import static com.xebialabs.deployit.plugin.overthere.DefaultExecutionOutputHandler.handleStderr;
import static com.xebialabs.deployit.plugin.overthere.DefaultExecutionOutputHandler.handleStdout;
import static com.xebialabs.deployit.plugin.remoting.preview.PreviewOverthereConnection.getPreviewConnection;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.MDC_KEY_SCRIPT_PATH;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.appendScriptDir;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.appendScripts;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.dumpScript;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.loadScript;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.uploadScript;
import static com.xebialabs.overthere.OperatingSystemFamily.WINDOWS;
import static com.xebialabs.overthere.util.OverthereUtils.getName;

@SuppressWarnings("serial")
abstract class PowerShellStep implements PreviewStep {

    private static final String BASE_RUNTIME_PATH = "powershell/runtime";
    private static final String DEFAULT_POWER_SHELL_PATH = "powershell";

    private HostContainer container;
    private Deployed<?, ?> deployed;
    private String scriptPath;
    private Map<String, Object> powerShellVars;
    private String description;
    private boolean uploadArtifactData = true;
    private Preview preview;

    public PowerShellStep(HostContainer container, Deployed<?, ?> deployed, String scriptPath, Map<String, Object> powerShellVars, String description) {
        this.container = checkNotNull(container, "container is null");
        checkArgument(this.container.getHost().getOs() == WINDOWS, "PowerShell scripts can only be run on Windows hosts");
        this.deployed = deployed;
        this.scriptPath = checkNotNull(scriptPath, "scriptPath is null");
        this.powerShellVars = checkNotNull(powerShellVars, "powerShellVars is null");
        this.description = checkNotNull(description, "description is null");
    }

    protected StepExitCode doExecute(ExecutionContext ctx) throws Exception {
        MDC.put(MDC_KEY_SCRIPT_PATH, scriptPath);
        try {
            OverthereConnection conn = container.getHost().getConnection();
            try {
                String generatedScript = aggregateScript(conn);
                dumpScript(getName(scriptPath), generatedScript, scriptsLogger);
                OverthereFile uploadedScriptFile = uploadScript(conn, scriptPath, generatedScript);

                int res = executePowerShellScript(ctx, conn, uploadedScriptFile);

                logger.debug("Exit code: {}", res);
                if(res == 0) {
                    return StepExitCode.SUCCESS;
                } else {
                    return StepExitCode.FAIL;
                }
            } finally {
                Closeables.closeQuietly(conn);
            }
        } finally {
            MDC.remove(MDC_KEY_SCRIPT_PATH);
        }
    }

    @Override
    public Preview getPreview() {
        if(preview == null) {
            OverthereConnection poc = getPreviewConnection();
            try {
                String contents = aggregateScript(poc);
                preview = Preview.withSourcePathAndContents(scriptPath, contents);
            } finally {
                poc.close();
            }
        }
        return preview;
    }

    protected String aggregateScript(OverthereConnection connection) {
        StringBuilder b = new StringBuilder();
        appendBaseRuntimeScript(b);
        appendContainerRuntimeScripts(b);
        appendContainerLibraryScripts(b);
        appendDeployedLibraryScripts(b);
        appendPowerShellVars(connection, b);
        appendScript(b);
        // Replaces all LF's that are NOT preceded by a CR with CRLF.
        return b.toString().replaceAll("(?<!\r)\n", "\r\n");
    }

    private void appendBaseRuntimeScript(StringBuilder b) {
        appendScriptDir(BASE_RUNTIME_PATH, b);
    }

    private void appendContainerRuntimeScripts(StringBuilder b) {
        if(container.hasProperty("runtimePath")) {
            String runtimePath = container.getProperty("runtimePath");
            appendScriptDir(runtimePath, b);
        }
    }

    private void appendContainerLibraryScripts(StringBuilder b) {
        if(container instanceof PowerShellContainer) {
            appendScripts(((PowerShellContainer) container).getLibraryScripts(), b);
        }
    }

    private void appendDeployedLibraryScripts(StringBuilder b) {
        if (deployed instanceof BasePowerShellDeployed) {
            appendScripts(((BasePowerShellDeployed<?, ?>) deployed).getLibraryScripts(), b);
        }
    }

    private void appendPowerShellVars(OverthereConnection connection, StringBuilder b) {
        b.append("# PowerShellVars\n");
        b.append(PowerShellVarsConverter.javaToPowerShell(connection, powerShellVars, uploadArtifactData));
    }

    private void appendScript(StringBuilder b) {
        b.append(loadScript(scriptPath));
    }

    private int executePowerShellScript(ExecutionContext ctx, OverthereConnection conn, OverthereFile script) {
        logger.info("Executing PowerShell script {} on {}", script, conn);
        DefaultExecutionOutputHandler stdoutHandler = handleStdout(ctx);
        DefaultExecutionOutputHandler stderrHandler = handleStderr(ctx);
        try {
            CmdLine cmdLine;
            if(container instanceof PowerShellContainer) {
                cmdLine = ((PowerShellContainer) container).getScriptCommandLine(script);
            } else {
                cmdLine = getDefaultScriptCommandLine(DEFAULT_POWER_SHELL_PATH, script);
            }
            return conn.execute(stdoutHandler, stderrHandler, cmdLine);
        } finally {
            closeQuietly(stdoutHandler);
            closeQuietly(stderrHandler);
        }
    }

    @Override
    public String getDescription() {
        return description;
    }

    public void setUploadArtifactData(boolean uploadArtifactData) {
        this.uploadArtifactData = uploadArtifactData;
    }

    public static CmdLine getDefaultScriptCommandLine(String powerShellPath, OverthereFile script) {
        return CmdLine.build(powerShellPath, "-ExecutionPolicy", "Unrestricted", "-Inputformat", "None", "-NonInteractive", "-NoProfile",
            "-Command", "$ErrorActionPreference = 'Stop'; & " + script.getPath() + "; if($LastExitCode) { Exit $LastExitCode; }");
    }

    private static Logger logger = LoggerFactory.getLogger(PowerShellStep.class);

    private static final Logger scriptsLogger = LoggerFactory.getLogger("com.xebialabs.deployit.plugin.powershell.scripts");

}
