package com.xebialabs.deployit.plugin.powershell;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.base.Function;
import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.inspection.Inspect;
import com.xebialabs.deployit.plugin.api.inspection.InspectionContext;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseContainer;
import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.overthere.CmdLine;
import com.xebialabs.overthere.OverthereFile;

import static com.xebialabs.deployit.plugin.powershell.PowerShellStepUtils.getDefaultScriptCommandLine;

import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Sets.newHashSet;

@SuppressWarnings("serial")
@Metadata(virtual = true)
public class BasePowerShellContainer extends BaseContainer implements PowerShellContainer {

    private static final String STANDARD_RUNTIME_LIBS = "powershell/runtime/base.ps1";

    @Property(asContainment = true, description = "Host upon which the container resides")
    private Host host;

    @Property(required = false, hidden = true, defaultValue = "powershell", description = "")
    private String powerShellPath = "powershell";

    @Property(required = false, defaultValue = STANDARD_RUNTIME_LIBS, hidden = true, description = "List of scripts to append to the the step script.")
    private List<String> libraryScripts;

    @Property(hidden = true, required = false)
    private Set<String> deployedsToDiscover = newHashSet();

    @Property(hidden = true, defaultValue = "0", description = "Order at which the discover step will be executed.")
    private int discoverOrder;

    @Property(hidden = true, required = false, description = "Script invoked to inspect this container and discover deployeds on it.")
    private String discoverScript;

    @Override
    public Host getHost() {
        return host;
    }

    public void setHost(Host host) {
        this.host = host;
    }

    public String getPowerShellPath() {
        return powerShellPath;
    }

    public void setPowerShellPath(String powerShellPath) {
        this.powerShellPath = powerShellPath;
    }

    public Set<String> getDeployedsToDiscover() {
        return deployedsToDiscover;
    }

    public void setDeployedsToDiscover(HashSet<String> deployedsToDiscover) {
        this.deployedsToDiscover = deployedsToDiscover;
    }

    public int getDiscoverOrder() {
        return discoverOrder;
    }

    public void setDiscoverOrder(int discoverOrder) {
        this.discoverOrder = discoverOrder;
    }

    public String getDiscoverScript() {
        return discoverScript;
    }

    public void setDiscoverScript(String discoverScript) {
        this.discoverScript = discoverScript;
    }

    @Override
    public CmdLine getScriptCommandLine(OverthereFile script) {
        return getDefaultScriptCommandLine(powerShellPath, script);
    }

    @Override
    public List<String> getLibraryScripts() {
        return libraryScripts;
    }

    public List<Step> controlTaskDispatch(String name, Map<String, String> args) {
        return ControlTaskDelegate.dispatch(name, args, this, this);
    }

    @Inspect
    public void inspect(InspectionContext ctx) {
        if (!Strings.isNullOrEmpty(discoverScript)) {
            final Map<String, Object> varsCtx = newHashMap();
            varsCtx.put("container", this);
            ctx.addStep(new PowerShellInspectionStep(this, null, discoverScript, varsCtx,"Inspect properties of " + this));
        }

        Iterable<Type> typesToDiscovery = Iterables.transform(deployedsToDiscover, new Function<String, Type>() {
            @Override
            public Type apply(String input) {
                return Type.valueOf(input);
            }
        });

        PowerShellDeployedInspectionHelper inspectionHelper = new PowerShellDeployedInspectionHelper();
        inspectionHelper.discoverDeployeds(this, ctx, Lists.newArrayList(typesToDiscovery));
    }

}
