package com.xebialabs.deployit.plugin.powershell;

import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;

import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseDeployed;
import com.xebialabs.deployit.plugin.overthere.HostContainer;

import static com.google.common.collect.Sets.newHashSet;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;

@SuppressWarnings("serial")
@Metadata(virtual = true)
public class BasePowerShellDeployed<D extends Deployable, C extends HostContainer> extends BaseDeployed<D, C> {

    public static final String STEP_OPTION_NONE = "none";

    public static final String STEP_OPTION_UPLOAD_ARTIFACT_DATA = "uploadArtifactData";

    public static final String STEP_OPTION_UPLOAD_CLASSPATH_RESOURCES = "uploadClasspathResources";

    public static final String STEP_OPTION_EXPOSE_DEPLOYED_APPLICATION = "exposeDeployedApplication";

    public static final String STEP_OPTION_EXPOSE_PREVIOUS_DEPLOYED = "exposePreviousDeployed";

    public static final String STEP_OPTIONS_REGEX = "(" + STEP_OPTION_NONE + "|" + STEP_OPTION_UPLOAD_ARTIFACT_DATA + "|"
        + STEP_OPTION_UPLOAD_CLASSPATH_RESOURCES + "|" + STEP_OPTION_EXPOSE_DEPLOYED_APPLICATION + "|" + STEP_OPTION_EXPOSE_PREVIOUS_DEPLOYED + ")";

    public static final String STEP_OPTIONS_DESCRIPTION_SUFFIX = " (1 or more of: " + STEP_OPTION_NONE + "," + STEP_OPTION_UPLOAD_ARTIFACT_DATA + ","
        + STEP_OPTION_UPLOAD_CLASSPATH_RESOURCES + "," + STEP_OPTION_EXPOSE_DEPLOYED_APPLICATION + "," + STEP_OPTION_EXPOSE_PREVIOUS_DEPLOYED + ").";

    @Property(required = false, hidden = true, description = "List of scripts to append to the the step script.")
    private List<String> libraryScripts = newArrayList();

    @Property(hidden = true, required = false, description = "Additional classpath resources that should be uploaded to the working directory before executing the script.")
    private Set<String> classpathResources = newHashSet();

    @Property(hidden = true, defaultValue = "50", description = "Order at which the discover step will be executed.")
    private int discoverOrder;

    @Property(hidden = true, required = false, description = "Script invoked to discover and inspect deployeds of this type.")
    private String discoverScript;

    public boolean addStep(DeploymentPlanningContext ctx, Delta delta, int order, String scriptPath, String verb, Set<String> options) {
        return addStep(ctx, delta, null, order, scriptPath, verb, options);
    }

    public boolean addStepWithCheckpoint(DeploymentPlanningContext ctx, Delta delta, Operation operation, int order, String scriptPath, String verb, Set<String> options) {
        Preconditions.checkNotNull(operation, "operation is null");
        return addStep(ctx, delta, operation, order, scriptPath, verb, options);
    }

    protected boolean addStep(DeploymentPlanningContext ctx, Delta delta, Operation operation, int order, String scriptPath, String verb, Set<String> options) {
        if (!Strings.isNullOrEmpty(scriptPath)) {
            PowerShellDeploymentStep step = createStep(ctx, delta, order, scriptPath, verb, options);

            if(operation == null) {
                ctx.addStep(step);
            } else {
                ctx.addStepWithCheckpoint(step, delta, operation);
            }

            return true;
        } else {
            return false;
        }
    }

    PowerShellDeploymentStep createStep(DeploymentPlanningContext ctx, Delta delta, int order, String scriptPath, String verb, Set<String> options) {
        Map<String, Object> vars = getVars(delta.getPrevious(), ctx.getDeployedApplication(), options);
        String description = getDescription(verb);
        PowerShellDeploymentStep step = new PowerShellDeploymentStep(order, getContainer(), this, scriptPath, vars, description);
        step.setUploadArtifactData(options.contains(STEP_OPTION_UPLOAD_ARTIFACT_DATA));
        if (options.contains(STEP_OPTION_UPLOAD_CLASSPATH_RESOURCES)) {
            step.setClasspathResources(newArrayList(classpathResources));
        }
        return step;
    }

    protected Map<String, Object> getVars(Deployed<?, ?> previous, DeployedApplication deployedApplication, Set<String> options) {
        Map<String, Object> vars = newHashMap();
        vars.put("deployed", this);
        if (options.contains(STEP_OPTION_EXPOSE_PREVIOUS_DEPLOYED)) {
            checkNotNull(previous, "previousDeployed is null");
            vars.put("previousDeployed", previous);
        }
        if (options.contains(STEP_OPTION_EXPOSE_DEPLOYED_APPLICATION)) {
            checkNotNull(deployedApplication, "deployedApplication is null");
            vars.put("deployedApplication", deployedApplication);
        }
        return vars;
    }

    protected String getDescription(String verb) {
        return String.format("%s %s on %s", verb, getName(), getContainer().getName());
    }

    public List<Step> controlTaskDispatch(String name, Map<String,String> args) {
        return ControlTaskDelegate.dispatch(name, args, this, getContainer());
    }

    public List<String> getLibraryScripts() {
        return libraryScripts;
    }

    public Set<String> getClasspathResources() {
        return classpathResources;
    }

    public int getDiscoverOrder() {
        return discoverOrder;
    }

    public String getDiscoverScript() {
        return discoverScript;
    }

}
