/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.plugin.stitch.service.engine.context;

import com.fasterxml.jackson.databind.JsonNode;
import com.xebialabs.xlplatform.documentation.PublicApi;

import java.util.List;

/**
 * InvocationDocument is abstraction for original input document for the executed Stitch Invocation.
 * It is used for extracting single or list of values based on pathExpressions.
 * <p>
 * Actual sub-class is {@link JsonInputContext}.
 * JsonInputContext is used for Json/YAML inputs. If YAML was in original input it is transformed into equivalent Json document.
 * Extraction of values and lists is done by calling methods with valid JsonPath expression
 * (see referral implementation of jayway.jsonPath:json-path)
 * <p>
 */
@PublicApi
public interface InputContext {

    /**
     * Returns true if path exists and has assigned value to it.
     * @param pathExpression on which we run check
     * @return evaluated flag for given {@code pathExpression}
     */
    boolean pathExistsWithValue(String pathExpression);

    /**
     * Returns {@link String} value in given path. If path doesn't exist or isn't single value {@code null} will be returned.
     * @param pathExpression on which we run check
     * @return evaluated {@link String} value for given {@code pathExpression}
     */
    String getStringValue(String pathExpression);

    /**
     * Returns {@link Boolean} value in given path. If path doesn't exist or isn't single value {@code null} will be returned.
     * @param pathExpression on which we run check
     * @return evaluated @link Boolean} value for given {@code pathExpression}
     */
    Boolean getBooleanValue(String pathExpression);

    /**
     * Returns {@link Integer} value in given path. If path doesn't exist or isn't single value {@code null} will be returned.
     * @param pathExpression on which we run check
     * @return evaluated {@link Integer} value for given {@code pathExpression}
     */
    Integer getIntegerValue(String pathExpression);

    /**
     * Returns {@link Long} value in given path. If path doesn't exist or isn't single value {@code null} will be returned.
     * @param pathExpression on which we run check
     * @return evaluated {@link Long} value for given {@code pathExpression}
     */
    Long getLongValue(String pathExpression);

    /**
     * Returns {@link Double} value in given path. If path doesn't exist or isn't single value {@code null} will be returned.
     * @param pathExpression on which we run check
     * @return evaluated {@link Double} value for given {@code pathExpression}
     */
    Double getDoubleValue(String pathExpression);

    /**
     * Returns {@link JsonNode} value in given path. If path doesn't exist {@code null} will be returned.
     * @param pathExpression on which we run check
     * @return evaluated {@link JsonNode} value for given {@code pathExpression}
     * @see JsonNode
     */
    JsonNode getJsonNode(String pathExpression);

    /**
     * Returns {@link List<String>} for given path. If path doesn't exist {@code null} will be returned.
     * @param pathExpression on which we run check
     * @return evaluated {@link List<String>} value for given {@code pathExpression}
     */
    List<String> getStringList(String pathExpression);

    /**
     * Returns {@link List<Boolean>} for given path. If path doesn't exist {@code null} will be returned.
     * @param pathExpression on which we run check
     * @return evaluated {@link List<Boolean>} value for given {@code pathExpression}
     */
    List<Boolean> getBooleanList(String pathExpression);

    /**
     * Returns {@link List<Integer>} for given path. If path doesn't exist {@code null} will be returned.
     * @param pathExpression on which we run check
     * @return evaluated {@link List<Integer>} value for given {@code pathExpression}
     */
    List<Integer> getIntegerList(String pathExpression);

    /**
     * Returns {@link List<Long>} for given path. If path doesn't exist {@code null} will be returned.
     * @param pathExpression on which we run check
     * @return evaluated {@link List<Long>} value for given {@code pathExpression}
     */
    List<Long> getLongList(String pathExpression);

    /**
     * Returns {@link List<Double>} for given path. If path doesn't exist {@code null} will be returned.
     * @param pathExpression on which we run check
     * @return evaluated {@link List<Double>} value for given {@code pathExpression}
     */
    List<Double> getDoubleList(String pathExpression);

    /**
     * Returns {@link List<JsonNode>} for given path. If path doesn't exist {@code null} will be returned.
     * @param pathExpression on which we run check
     * @return evaluated {@link List<JsonNode>} value for given {@code pathExpression}
     * @see JsonNode
     */
    List<JsonNode> getJsonNodeList(String pathExpression);

}
