package com.xebialabs.deployit.plugin.tomcat.contributor;

import java.util.List;
import com.google.common.base.Function;
import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.Iterables;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Multimaps;

import com.xebialabs.deployit.plugin.api.deployment.planning.Contributor;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.generic.step.ArtifactDeleteStep;
import com.xebialabs.deployit.plugin.overthere.HostContainer;
import com.xebialabs.deployit.plugin.tomcat.deployed.ContextElement;
import com.xebialabs.deployit.plugin.tomcat.deployed.ContextWarModule;

public class ContextXmlCleanup {

    @Contributor
    public void cleanupContextXml(final Deltas deltas, final DeploymentPlanningContext result) {

        Iterable<Delta> deltasTargetedToVirtualHosts = findDeltasTargetedToVirtualHosts(deltas);
        ListMultimap<String, Delta> deltasPerWebApp = groupByWebContext(deltasTargetedToVirtualHosts);

        for (String context : deltasPerWebApp.keySet()) {
            List<Delta> webAppDeltas = deltasPerWebApp.get(context);
            boolean undeploymentInProgress = undeploymentInProgress(webAppDeltas);

            Optional<Delta> optional = findWarModule(webAppDeltas);

            if (optional.isPresent()) {
                Delta warDelta = optional.get();
                ContextWarModule warModule = (ContextWarModule) getDeployed(warDelta);
                if (undeploymentInProgress) {
                    if (warModule.isManageContextXml()) {
                        result.addStep(createDeleteStep((HostContainer) warModule.getContainer(), warModule.getDestroyOrderOfContextXml()+1,
                                                        warModule.getContextXmlTargetDirectory(), context));
                    }
                } else if (warDelta.getOperation() == Operation.DESTROY) {
                    throw new RuntimeException("There are still resources referencing war. War can only be undeployed when associated resources are also undeployed.");
                }
            }
        }
    }

    private Optional<Delta> findWarModule(final List<Delta> webAppDeltas) {
        return Iterables.tryFind(webAppDeltas, new Predicate<Delta>() {
            @Override
            public boolean apply(Delta input) {
                Deployed deployed = getDeployed(input);
                return deployed instanceof ContextWarModule;
            }
        });
    }

    private boolean undeploymentInProgress(final List<Delta> webAppDeltas) {
        return Iterables.all(webAppDeltas, new Predicate<Delta>() {
            @Override
            public boolean apply(Delta input) {
                return input.getOperation() == Operation.DESTROY;
            }
        });
    }

    private Deployed getDeployed(Delta delta) {
        return delta.getOperation() == Operation.DESTROY ? delta.getPrevious() : delta.getDeployed();
    }

    private Iterable<Delta> findDeltasTargetedToVirtualHosts(Deltas deltas) {
        final Type APPLICATION_CONTEXT_TYPE = Type.valueOf("tomcat.VirtualHost");
        return FluentIterable.from(deltas.getDeltas()).filter(new Predicate<Delta>() {
            @Override
            public boolean apply(Delta input) {
                Deployed deployed = getDeployed(input);
                if (deployed.getContainer().getType().getDescriptor().isAssignableTo(APPLICATION_CONTEXT_TYPE)) {
                    return deployed instanceof ContextElement || deployed instanceof ContextWarModule;
                }
                return false;
            }
        });
    }

    private ListMultimap<String, Delta> groupByWebContext(Iterable<Delta> deltas) {
        return Multimaps.index(deltas, new Function<Delta, String>() {
            @Override
            public String apply(Delta input) {
                Deployed deployed = getDeployed(input);
                String context;
                if (deployed instanceof ContextElement) {
                    context = ((ContextElement)deployed).getContext();
                } else {
                    context = ((ContextWarModule)deployed).getContextRoot();
                }
                return context;
            }
        });
    }

    private ArtifactDeleteStep createDeleteStep(HostContainer container, int order,String targetDirectory, String context) {
        ArtifactDeleteStep step = new ArtifactDeleteStep(order, container, targetDirectory);
        step.setTargetFile(context + ".xml");
        step.setDescription("Removing "+ context +" xml from " + targetDirectory);
        return step;
    }

}
