package com.xebialabs.deployit.plugin.tomcat.contributor;

import java.util.List;
import com.google.common.base.Function;
import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.Iterables;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Multimaps;

import com.xebialabs.deployit.plugin.api.deployment.planning.Contributor;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.generic.step.ArtifactDeleteStep;
import com.xebialabs.deployit.plugin.overthere.HostContainer;
import com.xebialabs.deployit.plugin.tomcat.deployed.ContextElement;
import com.xebialabs.deployit.plugin.tomcat.deployed.ContextWarModule;

public class ContextXmlCleanup {

    @Contributor
    public static void cleanupContextXml(final Deltas deltas, final DeploymentPlanningContext result) {

        Iterable<Delta> deltasTargetedToVirtualHosts = findDeltasTargetedToVirtualHosts(deltas);

        ListMultimap<CleanupKey, Delta> deltasPerWebApp = groupByVirtualHostAndWebContext(deltasTargetedToVirtualHosts);

        for (CleanupKey key : deltasPerWebApp.keySet()) {
            List<Delta> webAppDeltas = deltasPerWebApp.get(key);
            boolean undeploymentInProgress = undeploymentInProgress(webAppDeltas);

            Optional<Delta> optional = findWarModule(webAppDeltas);

            if (optional.isPresent()) {
                Delta warDelta = optional.get();
                ContextWarModule<?> warModule = (ContextWarModule<?>) getDeployed(warDelta);
                if (undeploymentInProgress) {
                    if (warModule.isManageContextXml()) {
                        result.addStep(createDeleteStep(warModule.getContainer(), warModule.getDestroyOrderOfContextXml() + 1,
                                warModule.getContextXmlTargetDirectory(), key.getContext()));
                    }
                } else if (warDelta.getOperation() == Operation.DESTROY) {
                    throw new RuntimeException("There are still resources referencing the WAR. A WAR can only be undeployed when associated resources are also undeployed.");
                }
            }
        }
    }

    private static Optional<Delta> findWarModule(final List<Delta> webAppDeltas) {
        return Iterables.tryFind(webAppDeltas, new Predicate<Delta>() {
            @Override
            public boolean apply(Delta input) {
                Deployed<?, ?> deployed = getDeployed(input);
                return deployed instanceof ContextWarModule;
            }
        });
    }

    private static boolean undeploymentInProgress(final List<Delta> webAppDeltas) {
        return Iterables.all(webAppDeltas, new Predicate<Delta>() {
            @Override
            public boolean apply(Delta input) {
                return input.getOperation() == Operation.DESTROY;
            }
        });
    }

    private static Deployed<?, ?> getDeployed(Delta delta) {
        return delta.getOperation() == Operation.DESTROY ? delta.getPrevious() : delta.getDeployed();
    }

    private static Iterable<Delta> findDeltasTargetedToVirtualHosts(Deltas deltas) {
        final Type APPLICATION_CONTEXT_TYPE = Type.valueOf("tomcat.VirtualHost");
        return FluentIterable.from(deltas.getDeltas()).filter(new Predicate<Delta>() {
            @Override
            public boolean apply(Delta input) {
                Deployed<?, ?> deployed = getDeployed(input);
                if (deployed.getContainer().getType().getDescriptor().isAssignableTo(APPLICATION_CONTEXT_TYPE)) {
                    return deployed instanceof ContextElement || deployed instanceof ContextWarModule;
                }
                return false;
            }
        });
    }

    private static ListMultimap<CleanupKey, Delta> groupByVirtualHostAndWebContext(final Iterable<Delta> deltas) {
        return Multimaps.index(deltas, new Function<Delta, CleanupKey>() {
            @Override
            public CleanupKey apply(Delta input) {
                Deployed<?, ?> deployed = getDeployed(input);
                String context;
                if (deployed instanceof ContextElement) {
                    context = ((ContextElement<?>) deployed).getContext();
                } else {
                    context = ((ContextWarModule<?>) deployed).getContextRoot();
                }
                return new CleanupKey(context, deployed.getContainer().getId());
            }
        });
    }

    private static ArtifactDeleteStep createDeleteStep(HostContainer container, int order, String targetDirectory, String context) {
        ArtifactDeleteStep step = new ArtifactDeleteStep(order, container, targetDirectory);
        step.setTargetFile(context + ".xml");
        step.setDescription("Removing " + context + " xml from " + targetDirectory);
        return step;
    }

    private static class CleanupKey {
        private final String context;
        private final String containerId;

        public CleanupKey(final String context, final String containerId) {
            this.context = context;
            this.containerId = containerId;
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            final CleanupKey that = (CleanupKey) o;

            if (containerId != null ? !containerId.equals(that.containerId) : that.containerId != null) return false;
            if (context != null ? !context.equals(that.context) : that.context != null) return false;

            return true;
        }

        @Override
        public int hashCode() {
            int result = context != null ? context.hashCode() : 0;
            result = 31 * result + (containerId != null ? containerId.hashCode() : 0);
            return result;
        }

        public String getContext() {
            return context;
        }

        public String getContainerId() {
            return containerId;
        }
    }

}
