package com.xebialabs.deployit.plugin.was.step;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.was.container.BaseCell;
import com.xebialabs.deployit.plugin.was.container.Cell;
import com.xebialabs.deployit.plugin.was.container.WasVersion;
import com.xebialabs.deployit.plugin.was.util.WasVersionInfoParser;
import com.xebialabs.overthere.CmdLine;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.util.CapturingOverthereProcessOutputHandler;
import com.xebialabs.xlplatform.satellite.Satellite;
import com.xebialabs.xlplatform.satellite.SatelliteAware;

import static com.xebialabs.overthere.util.CapturingOverthereProcessOutputHandler.capturingHandler;

@SuppressWarnings("serial")
public class GetWasVersionStep implements Step, SatelliteAware {

    private Cell cell;

    public GetWasVersionStep(Cell cell) {
        this.cell = cell;
    }

    @Override
    public int getOrder() {
        return 50;
    }

    @Override
    public StepExitCode execute(ExecutionContext ctx) throws Exception {
        OverthereConnection adminConnection = cell.getHost().getConnection();
        try {
            CapturingOverthereProcessOutputHandler capturedOutput = capturingHandler();
            int res = adminConnection.execute(capturedOutput, CmdLine.build(cell.getVersionInfoPath()));
            if (res != 0) {
                ctx.logError("Non-zero exitcode from " + cell.getVersionInfoPath() + ": " + res);
                return StepExitCode.FAIL;
            }
            return checkWasVersion(ctx, capturedOutput);
        } finally {
            adminConnection.close();
        }
    }

    private StepExitCode checkWasVersion(ExecutionContext ctx,    CapturingOverthereProcessOutputHandler capturedOutput) {
        WasVersionInfoParser wasVersionInfoParser = new WasVersionInfoParser(getVersionInfoNameField()).parse(capturedOutput.getOutputLines());

        if (wasVersionInfoParser.isSuccess()) {
            if (wasVersionInfoParser.majorVersion == 6 && wasVersionInfoParser.minorVersion == 1) {
                ((BaseCell) cell).setVersion(WasVersion.WAS_61);
            } else if (wasVersionInfoParser.majorVersion == 7 && wasVersionInfoParser.minorVersion == 0) {
                ((BaseCell) cell).setVersion(WasVersion.WAS_70);
            } else if (wasVersionInfoParser.majorVersion == 8 && wasVersionInfoParser.minorVersion == 0) {
                ((BaseCell) cell).setVersion(WasVersion.WAS_80);
            } else if (wasVersionInfoParser.majorVersion == 8 && wasVersionInfoParser.minorVersion == 5) {
                ((BaseCell) cell).setVersion(WasVersion.WAS_85);
            } else {
                ctx.logError("Cannot set version on " + cell + ": version " + wasVersionInfoParser.majorVersion + "." + wasVersionInfoParser.minorVersion + " is not supported");
                return StepExitCode.FAIL;
            }
            return StepExitCode.SUCCESS;
        }

        ctx.logError("Could not determine installed version of " + cell);
        return StepExitCode.FAIL;
    }

    private String getVersionInfoNameField() {
        return ((BaseCell) cell).getProperty("versionInfoNameField");
    }

    @Override
    public String getDescription() {
        return "Get version of " + cell;
    }

    @Override
    public Satellite getSatellite() {
        return cell.getManagingContainer().getHost().getSatellite();
    }

}
