/*
 * Copyright (c) 2008-2011 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.wls.runbook;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.google.common.collect.Sets;
import com.xebialabs.deployit.Change;
import com.xebialabs.deployit.ChangePlan;
import com.xebialabs.deployit.ChangeResolution;
import com.xebialabs.deployit.RunBook;
import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.ci.Deployment;
import com.xebialabs.deployit.plugin.wls.ci.WlsCluster;
import com.xebialabs.deployit.plugin.wls.ci.WlsServer;
import com.xebialabs.deployit.plugin.wls.mapper.EarToWlsClusterMapper;
import com.xebialabs.deployit.plugin.wls.mapper.EarToWlsServerMapper;
import com.xebialabs.deployit.plugin.wls.mapper.EjbJarToWlsClusterMapper;
import com.xebialabs.deployit.plugin.wls.mapper.EjbJarToWlsServerMapper;
import com.xebialabs.deployit.plugin.wls.mapper.WarToWlsClusterMapper;
import com.xebialabs.deployit.plugin.wls.mapper.WarToWlsServerMapper;
import com.xebialabs.deployit.plugin.wls.step.ActivatePendingChangesStep;
import com.xebialabs.deployit.plugin.wls.step.CreateWlsServerStep;
import com.xebialabs.deployit.plugin.wls.step.DestroyWlsServerStep;
import com.xebialabs.deployit.plugin.wls.step.ModifyWlsServerStep;
import com.xebialabs.deployit.plugin.wls.step.StartStopStepFactory;

/**
 * Supports the creation, modification and deletion of a {@link WlsServer Managed Server}.
 * 
 * <h4>Conditions</h4>
 * 
 * Will trigger if the change plan contains the addition, modification or deletion of a {@code WebLogicServer} CI.
 * 
 * <h4>Actions</h4>
 * 
 * Addition:
 * 
 * <ol>
 * <li>A default server is created
 * <li>The server is modified
 * </ol>
 * 
 * Modification:
 * 
 * <ol>
 * <li>The server is stopped
 * <li>The server is modified
 * <li>The server is started
 * </ol>
 * 
 * Deletion:
 * 
 * <ol>
 * <li>The server is stopped
 * <li>The server is deleted
 * </ol>
 */
public class WlsServerRunBook implements RunBook {

	@SuppressWarnings("unchecked")
	public Collection<ChangeResolution> resolve(ChangePlan changePlan) {
		Collection<ChangeResolution> changeResolutions = new ArrayList<ChangeResolution>();

		for (Change<?> change : changePlan.getChanges()) {
			if (change.getConfigurationItemClass().equals(WlsServer.class)) {
				final Change<WlsServer> serverChange = (Change<WlsServer>) change;
				if (serverChange.getNewRevision() != null && serverChange.getNewRevision().isRunningVersion8())
					continue;
				if (serverChange.getOldRevision() != null && serverChange.getOldRevision().isRunningVersion8())
					continue;

				if (change.isDeletion()) {
					changeResolutions.add(new ChangeResolution(change, getDeletionSteps(serverChange)));
				}
				if (change.isAddition()) {
					changeResolutions.add(new ChangeResolution(change, getCreationSteps(serverChange)));
				}
				if (change.isModification() && !modificationIsHandledByWlsDeploymentRunBook(changePlan, serverChange)) {
					changeResolutions.add(new ChangeResolution(change, getModificationSteps(serverChange)));
				}
			}
		}

		return changeResolutions;
	}

	private boolean modificationIsHandledByWlsDeploymentRunBook(ChangePlan changePlan, Change<WlsServer> serverChange) {
		for (Change<Deployment> deploymentChange : WlsServerRunBook.<Deployment> getChangesForCiType(changePlan, Deployment.class)) {

			if (deploymentChange.isModification()) {
				EarToWlsServerMapper earToServerMapper = new EarToWlsServerMapper(deploymentChange);
				EjbJarToWlsServerMapper ejbToServerMapper = new EjbJarToWlsServerMapper(deploymentChange);
				WarToWlsServerMapper warToServerMapper = new WarToWlsServerMapper(deploymentChange);
				EarToWlsClusterMapper earToWlsClusterMapper = new EarToWlsClusterMapper(deploymentChange);
				WarToWlsClusterMapper warToWlsClusterMapper = new WarToWlsClusterMapper(deploymentChange);
				EjbJarToWlsClusterMapper ejbToWlsClusterMapper = new EjbJarToWlsClusterMapper(deploymentChange);

				Set<WlsServer> oldTargetServers = new HashSet<WlsServer>();
				oldTargetServers.addAll(earToServerMapper.getOldTargets());
				oldTargetServers.addAll(warToServerMapper.getOldTargets());
				oldTargetServers.addAll(ejbToServerMapper.getOldTargets());

				Set<WlsCluster> oldTargetClusters = new HashSet<WlsCluster>();
				oldTargetClusters.addAll(earToWlsClusterMapper.getOldTargets());
				oldTargetClusters.addAll(warToWlsClusterMapper.getOldTargets());
				oldTargetClusters.addAll(ejbToWlsClusterMapper.getOldTargets());
				for (WlsCluster cluster : oldTargetClusters) {
					oldTargetServers.addAll(cluster.getServers());
				}

				Set<WlsServer> newTargetServers = new HashSet<WlsServer>();
				newTargetServers.addAll(earToServerMapper.getNewTargets());
				newTargetServers.addAll(warToServerMapper.getNewTargets());
				newTargetServers.addAll(ejbToServerMapper.getNewTargets());

				Set<WlsCluster> newTargetClusters = new HashSet<WlsCluster>();
				newTargetClusters.addAll(earToWlsClusterMapper.getNewTargets());
				newTargetClusters.addAll(warToWlsClusterMapper.getNewTargets());
				newTargetClusters.addAll(ejbToWlsClusterMapper.getNewTargets());
				for (WlsCluster cluster : newTargetClusters) {
					newTargetServers.addAll(cluster.getServers());
				}

				if (oldTargetServers.contains(serverChange.getOldRevision()) && newTargetServers.contains(serverChange.getNewRevision())) {
					return true;
				}
			}
		}
		return false;
	}

	@SuppressWarnings("unchecked")
	private static <T extends Serializable> Set<Change<T>> getChangesForCiType(ChangePlan cp, Class<? extends Serializable> changeTypeClass) {
		Set<Change<T>> changesOfType = Sets.newHashSet();
		for (Change<?> change : cp.getChanges()) {
			if (changeTypeClass.equals(change.getConfigurationItemClass())) {
				changesOfType.add((Change<T>) change);
			}
		}
		return changesOfType;
	}

	public List<Step> getCreationSteps(Change<WlsServer> change) {
		WlsServer server = change.getNewRevision();

		if (server.isRunningVersion8())
			return Collections.emptyList();

		List<Step> allSteps = new ArrayList<Step>();
		allSteps.add(new CreateWlsServerStep(server));
		allSteps.add(new ModifyWlsServerStep(server));
		allSteps.add(new ActivatePendingChangesStep(server.getDomain()));
		allSteps.add(new StartStopStepFactory().getStartStep(server));

		return allSteps;
	}

	public List<Step> getDeletionSteps(Change<WlsServer> change) {
		WlsServer server = change.getOldRevision();

		if (server.isRunningVersion8())
			return Collections.emptyList();

		List<Step> allSteps = new ArrayList<Step>();
		allSteps.add(new StartStopStepFactory().getStopStep(server));
		allSteps.add(new DestroyWlsServerStep(server));
		allSteps.add(new ActivatePendingChangesStep(server.getDomain()));
		return allSteps;
	}

	public List<Step> getModificationSteps(Change<WlsServer> change) {
		WlsServer server = change.getNewRevision();

		if (server.isRunningVersion8())
			return Collections.emptyList();

		List<Step> allSteps = new ArrayList<Step>();
		final StartStopStepFactory factory = new StartStopStepFactory();
		allSteps.add(factory.getStopStep(server));
		allSteps.add(new ModifyWlsServerStep(server));
		allSteps.add(new ActivatePendingChangesStep(server.getDomain()));
		allSteps.add(factory.getStartStep(server));
		return allSteps;
	}

}
