package com.xebialabs.deployit.plugin.wls.mapper;

import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;
import com.google.common.collect.Ordering;
import com.xebialabs.deployit.Change;
import com.xebialabs.deployit.ResolutionException;
import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.ci.Deployment;
import com.xebialabs.deployit.ci.DeploymentPackage;
import com.xebialabs.deployit.ci.Host;
import com.xebialabs.deployit.ci.artifact.NamedDeployableArtifact;
import com.xebialabs.deployit.ci.artifact.mapping.DeployableArtifactMapping;
import com.xebialabs.deployit.ci.artifact.mapping.PlaceholderFormat;
import com.xebialabs.deployit.ci.mapping.KeyValuePair;
import com.xebialabs.deployit.mapper.Pair;
import com.xebialabs.deployit.mapper.StepGeneratingMapper;
import com.xebialabs.deployit.plugin.wls.ci.*;
import com.xebialabs.deployit.plugin.wls.step.*;
import com.xebialabs.deployit.steps.CopyStep;
import org.apache.commons.lang.StringUtils;

import java.lang.reflect.Field;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Sets.newHashSet;
import static java.util.Collections.emptyList;
import static java.util.Collections.sort;

/**
 * WlsArtifactMapper....
 */
public class WlsArtifactMapper<S extends NamedDeployableArtifact, M extends DeployableArtifactMapping<? super S, ? super T>, T extends WlsTarget>
		extends StepGeneratingMapper<S, M, T> {

	protected List<Step> stageSteps = newArrayList();
	protected List<Step> noStageSteps = newArrayList();

	protected List<Step> startSteps = newArrayList();

	protected Set<CopyStep> copySteps = newHashSet();
	protected List<Step> retirementSteps = newArrayList();

	protected final String defaultOldVersion;
	protected final String defaultNewVersion;

	private long retirementTimeout = -1;

	private final Ordering<M> byDeploymentOrder = new Ordering<M>() {
		public int compare(M m1, M m2) {
			WlsArtifactMapping am1 = (WlsArtifactMapping) m1;
			WlsArtifactMapping am2 = (WlsArtifactMapping) m2;
			return am1.getDeploymentOrder() - am2.getDeploymentOrder();
		}
	};

	public WlsArtifactMapper(Change<Deployment> change) {
		super(change);
		defaultOldVersion = fetchDefaultVersion(change.getOldRevision());
		defaultNewVersion = fetchDefaultVersion(change.getNewRevision());
		//sort addedMapping by their deployment order to start the applications in an increasing sequence
		sort(addedMappings, byDeploymentOrder);
		//reverse sort deletedMappings by their deployment order to stop the applications in a decreasing sequence
		sort(deletedMappings, byDeploymentOrder.reverse());
	}

	private String fetchDefaultVersion(Deployment deployment) {
		if (deployment == null)
			return null;

		final DeploymentPackage aPackage = deployment.getSource();
		if (aPackage.getApplication() == null) {
			return "v-" + aPackage.getVersion();
		}
		String version = aPackage.getApplication().getLabel() + "-" + aPackage.getVersion();
		version = version.replace("Applications/", "");
		version = version.replace("/", "");
		return version;
	}

	@Override
	protected void generateAdditionStepsForAddedMapping(S newMappingSource, M newMapping, T newMappingTarget, List<Step> steps) {
		WlsArtifactMapping mapping = (WlsArtifactMapping) newMapping;
		if (!isNoStageMode(mapping)) {
			//stageMode
			stageSteps.addAll(generateStageModeDeploysSteps(mapping));
		} else {
			//noStageMode
			noStageSteps.addAll(generateNoStageModeDeploysSteps(mapping, copySteps));
		}

		startSteps.addAll(generateStartStep(mapping));

	}

	@Override
	protected void generateDeletionStepsForDeletedMapping(S oldMappingSource, M oldVersionOfModifiedMapping, T oldMappingTarget, List<Step> steps) {
		WlsArtifactMapping mapping = (WlsArtifactMapping) oldVersionOfModifiedMapping;

		final WlsTarget target = mapping.getTarget();
		final NamedDeployableArtifact artifact = mapping.getSource();

		if (target.isRunningVersion8()) {
			steps.add(new Wls8UndeployArtifactStep(target, artifact));
			return;
		}

		final String appVersion = getAppVersion(mapping, defaultOldVersion);
		switch (mapping.getDeploymentStrategy()) {
			case CLASSIC:
				if (mapping.isVersionnedArtifact()) {
					steps.add(new StopApplicationStep(target, artifact, appVersion));
					steps.add(new WlsUndeployArtifactStep(target, artifact, appVersion));
				} else {
					steps.add(new StopApplicationStep(target, artifact));
					steps.add(new WlsUndeployArtifactStep(target, artifact));
				}
				break;
			case STOP_START:
				if (mapping.isVersionnedArtifact()) {
					steps.add(new WlsUndeployArtifactStep(target, artifact, appVersion));
				} else {
					steps.add(new WlsUndeployArtifactStep(target, artifact));
				}
				break;
			case SIDE_BY_SIDE:
				retirementSteps.add(new StopApplicationStep(target, artifact, appVersion));
				retirementSteps.add(new WlsUndeployArtifactStep(target, artifact, appVersion));
				break;
		}
	}

	private String getAppVersion(WlsArtifactMapping mapping, String defaultVersion) {
		if (!mapping.isVersionnedArtifact()) {
			return null;
		}

		if (StringUtils.isBlank(mapping.getVersion())) {
			return defaultVersion;
		}
		return mapping.getVersion();
	}

	public void generateUndeploySteps(List<Step> steps) {
		generateDeletionSteps(steps);
	}

	public void generateDeploySteps(List<Step> steps) {
		generateAdditionSteps(steps);
		steps.addAll(stageSteps);
		steps.addAll(copySteps);
		steps.addAll(noStageSteps);
	}


	public void generateStartSteps(List<Step> steps) {
		steps.addAll(startSteps);

	}

	public void generateRetimentSteps(List<Step> steps) {
		if (retirementTimeout > 0 && !retirementSteps.isEmpty()) {
			steps.add(new WaitingStep(retirementTimeout));
		}
		steps.addAll(retirementSteps);
	}

	private Collection<? extends Step> generateStartStep(WlsArtifactMapping mapping) {
		List<Step> steps = newArrayList();
		final WlsTarget target = mapping.getTarget();
		if (target.isRunningVersion8())
			return emptyList();

		switch (mapping.getDeploymentStrategy()) {
			case CLASSIC:
			case SIDE_BY_SIDE:
				if (mapping.isVersionnedArtifact())
					steps.add(new StartApplicationStep(target, mapping.getSource(), getAppVersion(mapping, defaultNewVersion)));
				else
					steps.add(new StartApplicationStep(target, mapping.getSource()));
				break;
			case STOP_START:
				//done by restarting the server
				break;

		}
		return steps;
	}


	private List<? extends Step> generateNoStageModeDeploysSteps(WlsArtifactMapping m, Set<CopyStep> copySteps) {
		final List<Step> steps = newArrayList();
		final Map<String, String> propertyMap = KeyValuePair.toMap(m.getKeyValuePairs());
		final PlaceholderFormat placeholderFormat = m.getPlaceholderFormat();
		final WlsTarget target = m.getTarget();


		String remoteDestination = "";
		String remoteDeploymentPlanDestination = "";

		for (Host remoteHost : findHosts(target)) {
			String separator = remoteHost.getFileSeparator();
			if (StringUtils.isBlank(m.getStagingDirectory()))
				throw new IllegalArgumentException("With noStage Mode, a staging directory must be filled.");

			remoteDestination = m.getStagingDirectory() + separator + m.getSource().getName() + "." + getExtension(m.getSource());
			final CopyStep copyStepArtifact = new CopyStep(Host.getLocalHost(), m.getSource().getLocation(), remoteHost, remoteDestination, propertyMap, placeholderFormat);
			copyStepArtifact.setCreateToplevelDirectory(true);
			copySteps.add(copyStepArtifact);

			if (m.getDeploymentPlan() != null) {
				if (StringUtils.isBlank(m.getDeploymentPlanStagingDirectory()))
					throw new IllegalArgumentException("With a deployment plan, a deployment plan staging directory must be filled.");
				remoteDeploymentPlanDestination = m.getDeploymentPlanStagingDirectory() + separator + m.getSource().getName() + ".xml";
				final CopyStep copyStepPlan = new CopyStep(Host.getLocalHost(), m.getDeploymentPlan().getLocation(), remoteHost, remoteDeploymentPlanDestination, propertyMap, placeholderFormat);
				copyStepPlan.setCreateToplevelDirectory(true);
				copySteps.add(copyStepPlan);
			}
		}


		if (target.isRunningVersion8()) {
			steps.add(new Wls8DeployNoStageArtifactStep(target, m.getSource(), remoteDestination));
			return steps;
		}


		WlsDeployNoStageArtifactStep step = new WlsDeployNoStageArtifactStep(target, m.getSource(), remoteDestination);
		steps.add(step);

		if (m.getDeploymentPlan() != null) {
			step.setRemoteDeploymentPlanPath(remoteDeploymentPlanDestination);
		}
		if (m.isVersionnedArtifact()) {
			throw new ResolutionException("nostage mode does not support the versioned applications");
		}

		if (m.getDeploymentOrder() != 100) {
			if (m.isVersionnedArtifact())
				steps.add(new ModifyDeploymentOrderStep(target, m.getSource(), getAppVersion(m, defaultNewVersion), m.getDeploymentOrder()));
			else
				steps.add(new ModifyDeploymentOrderStep(target, m.getSource(), m.getDeploymentOrder()));
		}


		return steps;
	}

	private List<? extends Step> generateStageModeDeploysSteps(WlsArtifactMapping m) {
		final List<Step> steps = newArrayList();
		final Map<String, String> propertyMap = KeyValuePair.toMap(m.getKeyValuePairs());
		final PlaceholderFormat placeholderFormat = m.getPlaceholderFormat();
		final WlsTarget target = m.getTarget();

		if (target.isRunningVersion8()) {
			steps.add(new Wls8DeployArtifactStep(target, m.getSource(), propertyMap, placeholderFormat));
			return steps;
		}

		WlsDeployArtifactStep step = new WlsDeployArtifactStep(target, m.getSource(), propertyMap, placeholderFormat);
		steps.add(step);
		if (m.getDeploymentPlan() != null) {
			Host activeHost = target.getDomain().getActiveHost();
			String remoteDeploymentPlanDestination = m.getDeploymentPlanStagingDirectory() + activeHost + m.getSource().getName() + ".xml";
			if (StringUtils.isBlank(m.getDeploymentPlanStagingDirectory()))
				throw new IllegalArgumentException("With a deployment plan, a deployment plan staging directory must be filled.");

			step.setDeploymentPlan(m.getDeploymentPlan(), remoteDeploymentPlanDestination);
		}

		if (m.isVersionnedArtifact()) {
			step.setAppVersion(getAppVersion(m, defaultNewVersion));
		}

		if (m.getDeploymentStrategy().equals(WlsDeploymentStrategy.SIDE_BY_SIDE)) {
			if (!m.isVersionnedArtifact())
				throw new ResolutionException("In a Side by Side strategy mode, the version must be enabled");

			if (isModifiedMapping(m)) {
				final int artifactRetireTimeout = m.getRetireTimeout();
				step.setRetireTimeout(artifactRetireTimeout);
				if (artifactRetireTimeout > retirementTimeout)
					retirementTimeout = artifactRetireTimeout;
			}
		}

		if (m.getSource() instanceof WlsSharedLibraryJar) {
			step.setLibraryModule(true);
		}

		if (m.getDeploymentOrder() != 100) {
			if (m.isVersionnedArtifact())
				steps.add(new ModifyDeploymentOrderStep(target, m.getSource(), getAppVersion(m, defaultNewVersion), m.getDeploymentOrder()));
			else
				steps.add(new ModifyDeploymentOrderStep(target, m.getSource(), m.getDeploymentOrder()));
		}


		return steps;
	}

	private boolean isModifiedMapping(final WlsArtifactMapping m) {
		return Collections2.filter(modifiedMappings, new Predicate<Pair<M, M>>() {
			public boolean apply(Pair<M, M> input) {
				return input.getSecond().equals(m);
			}
		} ).size() > 0;
	}


	private String getExtension(NamedDeployableArtifact artifact) {
		if (artifact instanceof WlsSharedLibrary) {
			WlsSharedLibrary sharedLibrary = (WlsSharedLibrary) artifact;
			return sharedLibrary.getSharedLibraryType().toString().toLowerCase();
		}
		try {
			final Field field = artifact.getClass().getField("ARCHIVE_EXTENSION");
			return "." + field.get(artifact).toString();
		} catch (Exception e) {
			throw new RuntimeException("Unexpected artifact " + artifact, e);
		}
	}

	private Set<Host> findHosts(WlsTarget target) {
		Set<Host> hosts = newHashSet();
		if (target instanceof WlsServer) {
			WlsServer server = (WlsServer) target;
			hosts.add(server.getHost());
		}
		if (target instanceof WlsCluster) {
			WlsCluster cluster = (WlsCluster) target;
			for (WlsServer server : cluster.getServers())
				hosts.add(server.getHost());
		}

		return hosts;
	}

	private boolean isNoStageMode(WlsArtifactMapping artefactMapping) {
		final WlsTarget target = artefactMapping.getTarget();
		if (target instanceof WlsCluster) {
			return artefactMapping.getStageMode().equals(WlsStageMode.NoStage);
		}
		if (target instanceof WlsServer) {
			WlsServer server = (WlsServer) target;
			return server.getStageMode().equals(WlsStageMode.NoStage) || artefactMapping.getStageMode().equals(WlsStageMode.NoStage);
		}
		return artefactMapping.getStageMode().equals(WlsStageMode.NoStage);
	}

}
