/*
 * Copyright (c) 2008-2011 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.wls.container;

import static com.google.common.collect.Maps.newHashMap;

import java.util.Map;
import java.util.Set;

import org.apache.commons.lang.StringUtils;

import com.google.common.collect.Sets;
import com.xebialabs.deployit.plugin.api.inspection.Inspect;
import com.xebialabs.deployit.plugin.api.inspection.InspectionPlanningContext;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseContainer;
import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.deployit.plugin.python.PythonInspectionStep;
import com.xebialabs.deployit.plugin.python.PythonManagedContainer;
import com.xebialabs.deployit.plugin.python.PythonManagingContainer;
import com.xebialabs.overthere.CmdLine;
import com.xebialabs.overthere.OperatingSystemFamily;
import com.xebialabs.overthere.OverthereFile;

/**
 * A domain in a WebLogic WLS installation.
 *
 * @see Cluster
 * @see Server
 */
@SuppressWarnings("serial")
@Metadata(root = Metadata.ConfigurationItemRoot.INFRASTRUCTURE, description = "WebLogic Domain which is a collection of WebLogic Server instances that is managed by a single Administration Server")
public class Domain extends BaseContainer implements PythonManagingContainer, PythonManagedContainer{

	@Property(required = true, label = "Host", description = "The host that runs the admin server")
	private Host host;

	@Property(required = true, defaultValue = "WEBLOGIC_10", label = "Version", description = "Version of Oracle WebLogic Server")
	private Version version;

	@Property(required = true, label = "WebLogic home", description = "The location of the WebLogic Server installation")
	private String wlHome;

	@Property(required = false, label = "WebLogic domain home", description = "The location of the WebLogic domain. Defaults to '<WebLogicHome>/../user_projects/domains/<Name>'")
	private String domainHome;

	@Property(required = true, defaultValue = "7001", label = "Administrative server port", description = "Port to be used by the AdminServer for this domain")
	private int port;

	@Property(required = true, label = "Administrative username", description = "Username which is used to login to the WebLogic Domain.")
	private String username;

	@Property(required = true, label = "Administrative password", password = true, description = "Password which is used to login to the WebLogic Domain.")
	private String password;

	@Property(required = true, defaultValue = "AdminServer", description = "The name of the admin server")
	private String adminServerName;

	@Property(required = true, defaultValue = "NodeManager", label = "Start Mode", description = "Tells how a managed server is start and stop, default is NodeManager, others are Script or Windows Service")
	private StartMode startMode;

	@Property(required = true, defaultValue = "true", hidden = true, label = "Run with daemon", description = "Set to true to execute commands with the Python daemon")
	private boolean runWithDaemon;
	
	@Property(description = "WebLogic clusters belonging to domain", asContainment = true)
	private Set<Cluster> clusters = Sets.newHashSet();

	@Override
    public PythonManagingContainer getManagingContainer() {
	    return this;
    }

	public String getWlstPath() {
		String fileSeparator = this.getHost().getOs().getFileSeparator();
		String wlstExtension;
		if (getHost().getOs() == OperatingSystemFamily.WINDOWS) {
			wlstExtension = ".cmd";
		} else {
			wlstExtension = ".sh";
		}
		return wlHome + fileSeparator + "common" + fileSeparator + "bin" + fileSeparator + "wlst" + wlstExtension;
	}

	@Override
	public CmdLine getScriptCommandLine(OverthereFile script) {
		CmdLine newCmdLine = new CmdLine();

		newCmdLine.addArgument(getWlstPath());
		newCmdLine.addArgument("-i");
		newCmdLine.addArgument(script.getPath());
		newCmdLine.addArgument(getUsername());
		newCmdLine.addArgument(getPassword());
		newCmdLine.addArgument(host.getSyntheticProperty("address") + ":" + port);

		return newCmdLine;
	}

	@Inspect
	public void inspect(InspectionPlanningContext ctx) {
		Map<String, Object> pythonVars = newHashMap();
		pythonVars.put("container", this);
		ctx.addStep(new PythonInspectionStep(this, this, "wls/container/inspect-domain.py", pythonVars, "Inspect " + this));
	}

	@Override
	public Host getHost() {
		return host;
	}

	public void setHost(Host activeHost) {
		this.host = activeHost;
	}


	public Version getVersion() {
    	return version;
    }

	public void setVersion(Version version) {
    	this.version = version;
    }

	public String getUsername() {
		return username;
	}

	public void setUsername(String username) {
		this.username = username;
	}

	public String getPassword() {
		return password;
	}

	public void setPassword(String password) {
		this.password = password;
	}

	public int getPort() {
		return port;
	}

	public void setPort(int port) {
		this.port = port;
	}

	public String getWlHome() {
		return wlHome;
	}

	public void setWlHome(String wlHome) {
		this.wlHome = wlHome;
	}

	public String getDomainHome() {
		return StringUtils.defaultIfEmpty(domainHome, getDefaultDomainHome());
	}

	private String getDefaultDomainHome() {
		final String fileSeparator = host.getOs().getFileSeparator();
		return wlHome.substring(0, wlHome.lastIndexOf(fileSeparator)) + fileSeparator + "user_projects" + fileSeparator + "domains" + fileSeparator + getName();
	}

	public void setDomainHome(String domainHome) {
		this.domainHome = domainHome;
	}

	public String getAdminServerName() {
		return adminServerName;
	}

	public void setAdminServerName(String adminServerName) {
		this.adminServerName = adminServerName;
	}

	public void setUseNodeManager(boolean useNodeManager) {
		this.startMode = StartMode.NodeManager;
	}

	public StartMode getStartMode() {
		return startMode;
	}

	public void setStartMode(StartMode startMode) {
		this.startMode = startMode;
	}

	public Set<Cluster> getClusters() {
    	return clusters;
    }

	public void addClusters(Cluster cluster) {
    	clusters.add(cluster);
    }
	

	@Override
	public String getRuntimePath() {
		return "wls/runtime";
	}

	@Override
	public boolean runWithDaemon() {
		return runWithDaemon;
	}

	public void setRunWithDaemon(boolean runWithDaemon) {
    	this.runWithDaemon = runWithDaemon;
    }

}
