package com.xebialabs.deployit.plugin.wls.deployed;

import com.google.common.base.Function;
import com.xebialabs.deployit.plugin.api.deployment.execution.DeploymentStep;
import com.xebialabs.deployit.plugin.api.deployment.planning.Create;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.planning.Destroy;
import com.xebialabs.deployit.plugin.api.deployment.planning.Modify;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.udm.DeployableArtifact;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifacts;
import com.xebialabs.deployit.plugin.api.udm.artifact.DerivedArtifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.PlaceholderReplacer;
import com.xebialabs.deployit.plugin.api.udm.base.BaseDeployed;
import com.xebialabs.deployit.plugin.api.validation.Placeholders;
import com.xebialabs.deployit.plugin.generic.freemarker.ConfigurationHolder;
import com.xebialabs.deployit.plugin.generic.step.ArtifactCopyStep;
import com.xebialabs.deployit.plugin.generic.step.ArtifactDeleteStep;
import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.deployit.plugin.wls.container.WlsContainer;
import com.xebialabs.overthere.OverthereFile;

import java.util.Collection;
import java.util.Collections;
import java.util.Map;

import static com.google.common.base.Strings.nullToEmpty;
import static com.google.common.collect.Collections2.transform;
import static com.google.common.collect.Maps.newHashMap;

@SuppressWarnings("serial")
@Metadata(virtual = true, description = "Base class for all deployeds meant to contain Applications")
@Placeholders
public class CopiedArtifact<D extends DeployableArtifact, C extends WlsContainer> extends BaseDeployed<D, C> implements DerivedArtifact<DeployableArtifact> {

    private Map<String, Object> freeMarkerContext = Collections.singletonMap("deployed", (Object) this);

    private OverthereFile placeholderProcessedFile;

    @Property(hidden = true, defaultValue = "50", description = "The order of the step in the step list for the create operation.")
    private int createOrder;

    @Property(hidden = true, defaultValue = "40", description = "The order of the step in the step list for the destroy operation.")
    private int destroyOrder;

    @Property(description = "Path to which artifact must be copied to on the wls server.")
    private String targetDirectory;

    @Property(defaultValue = "true", description = "Is the target directory shared by others on the wls server. When true, the target directory is not deleted during a destroy operation; only the artifacts copied to it.", required = false)
    private boolean targetDirectoryShared;

    @Property(defaultValue = "false", description = "Create the target directory on the wls server if it does not exist.", required = false)
    private boolean createTargetDirectory;

    @Property(required = false, description = "Name of the artifact on the wls server.")
    private String targetFile;

    @Property(defaultValue = "false", description = "Restart the target container", required = false)
    private boolean restartTarget;

    @Property(description = "A Map containing all the placeholders mapped to their values. Special values are &lt;ignore&gt; or &lt;empty&gt;", required = false)
    private Map<String, String> placeholders = newHashMap();

    @Create
    public void executeCreate(DeploymentPlanningContext ctx) {
        ctx.addSteps(createArtifactCopyStep());
    }

    protected Collection<DeploymentStep> createArtifactCopyStep() {
        final WlsContainer container = getContainer();
        return transform(container.getHosts(), new Function<Host, DeploymentStep>() {
            @Override
            public ArtifactCopyStep apply(Host input) {
                ArtifactCopyStep step = new ArtifactCopyStep(getCreateOrder(), getFile(), input, getTargetDirectory());
                step.setCreateTargetPath(isCreateTargetDirectory());
                step.setTargetFileName(resolveTargetFileName());
                step.setSourceFileDescription(getDeployable().getName());
                step.setDescription(getDescription("Copy"));
                return step;
            }
        });
    }

    @Destroy
    public void executeDestroy(DeploymentPlanningContext ctx) {
        ctx.addSteps(createArtifactDeleteStep());
    }

    protected Collection<DeploymentStep> createArtifactDeleteStep() {
        final WlsContainer container = getContainer();
        return transform(container.getHosts(), new Function<Host, DeploymentStep>() {
            @Override
            public ArtifactDeleteStep apply(Host input) {
                ArtifactDeleteStep step = new ArtifactDeleteStep(getDestroyOrder(), input, getDeployable(), getTargetDirectory());
                step.setTargetDirectoryShared(isTargetDirectoryShared());
                step.setTargetFile(resolveTargetFileName());
                step.setDescription(getDescription("Delete"));
                return step;
            }
        });
    }

    @Modify
    public void executeModify(DeploymentPlanningContext ctx, Delta d) {
        final CopiedArtifact<?, ?> old = (CopiedArtifact<?, ?>) d.getPrevious();
        old.executeDestroy(ctx);
        executeCreate(ctx);
    }

    public String getTargetDirectory() {
        return resolveExpression(targetDirectory);
    }

    public String getTargetFile() {
        return resolveExpression(targetFile);
    }

    public void setTargetFile(String targetFile) {
        this.targetFile = targetFile;
    }

    public boolean isCreateTargetDirectory() {
        return createTargetDirectory;
    }

    public void setCreateTargetDirectory(boolean createTargetDirectory) {
        this.createTargetDirectory = createTargetDirectory;
    }

    public boolean isTargetDirectoryShared() {
        return targetDirectoryShared;
    }

    public void setTargetDirectoryShared(boolean targetDirectoryShared) {
        this.targetDirectoryShared = targetDirectoryShared;
    }

    public String resolveExpression(String expression) {
        return ConfigurationHolder.resolveExpression(expression, getDeployedAsFreeMarkerContext());
    }

    public Map<String, Object> getDeployedAsFreeMarkerContext() {
        return freeMarkerContext;
    }

    protected String resolveTargetFileName() {
        return nullToEmpty(getTargetFile()).trim().isEmpty() ? getDeployable().getName() : getTargetFile();
    }

    public int getCreateOrder() {
        return createOrder;
    }

    public void setCreateOrder(int createOrder) {
        this.createOrder = createOrder;
    }

    public int getDestroyOrder() {
        return destroyOrder;
    }

    public void setDestroyOrder(int destroyOrder) {
        this.destroyOrder = destroyOrder;
    }

    @Override
    public OverthereFile getFile() {
        return placeholderProcessedFile;
    }

    @Override
    public void setFile(OverthereFile file) {
        this.placeholderProcessedFile = file;
    }

    @Override
    public DeployableArtifact getSourceArtifact() {
        return getDeployable();
    }

    @Override
    public Map<String, String> getPlaceholders() {
        return placeholders;
    }

    @Override
    public void setPlaceholders(Map<String, String> placeholders) {
        this.placeholders = placeholders;
    }

    @Override
    public void initFile(PlaceholderReplacer replacer) {
        Artifacts.replacePlaceholders(this, replacer);
    }

    public String getDescription(String verb) {
        return String.format("%s %s on %s", verb, getDeployable().getName(), getContainer().getName());
    }

}
