package com.xebialabs.deployit.plugin.wls.contributor;

import java.util.HashSet;
import java.util.List;
import java.util.Set;
import com.google.common.base.Function;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;

import com.xebialabs.deployit.plugin.api.deployment.planning.Contributor;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.base.BaseDeployableArchiveArtifact;
import com.xebialabs.deployit.plugin.wls.DeploymentStrategy;
import com.xebialabs.deployit.plugin.wls.ResourceTargetRestartPolicy;
import com.xebialabs.deployit.plugin.wls.container.Server;
import com.xebialabs.deployit.plugin.wls.container.WlsContainer;
import com.xebialabs.deployit.plugin.wls.deployed.CopiedArtifact;
import com.xebialabs.deployit.plugin.wls.deployed.ExtensibleDeployedArtifact;
import com.xebialabs.deployit.plugin.wls.deployed.Resource;
import com.xebialabs.deployit.plugin.wls.step.StartWlsServerWithScriptStep;
import com.xebialabs.deployit.plugin.wls.step.StartWlsTargetStep;
import com.xebialabs.deployit.plugin.wls.step.StopWlsTargetStep;

public class WlsContainerStopStartContributor {

    @Contributor
      public void stopContainers(Deltas deltas, DeploymentPlanningContext result) {
          for (WlsContainer target : gatherTargets(deltas.getDeltas())) {
              result.addSteps(target.getStopSteps(target.getStopOrder()));
          }
      }

    @Contributor
    public void startContainers(Deltas deltas, DeploymentPlanningContext result) {
        for (WlsContainer target : gatherTargets(deltas.getDeltas())) {
            result.addSteps(target.getStartSteps(target.getStartOrder()));
        }
    }

    private static Set<WlsContainer> gatherTargets(List<Delta> operations) {
        final Set<WlsContainer> wlsTargets = new HashSet<WlsContainer>();
        for (Delta operation : operations) {
            addTarget(wlsTargets, operation.getOperation(), operation.getDeployed());
            addTarget(wlsTargets, operation.getOperation(), operation.getPrevious());
        }
        return wlsTargets;
    }

    @SuppressWarnings("unchecked")
    private static void addTarget(Set<WlsContainer> wlsTargets, final Operation operation, Deployed<?, ?> deployed) {
        if (deployed == null)
            return;

        if (operation == Operation.NOOP)
            return;

        if (deployed instanceof Resource || deployed instanceof CopiedArtifact) {
            final boolean restartTarget = (Boolean) deployed.getProperty("restartTarget");
            ResourceTargetRestartPolicy restartPolicy = deployed.getProperty("targetRestartPolicy");
            if (restartTarget && restartPolicy == ResourceTargetRestartPolicy.STOP_START) {
                wlsTargets.add((WlsContainer) deployed.getContainer());
            }
        } else if (deployed instanceof ExtensibleDeployedArtifact) {
            ExtensibleDeployedArtifact<BaseDeployableArchiveArtifact> deployedArtifact = (ExtensibleDeployedArtifact<BaseDeployableArchiveArtifact>) deployed;
            final DeploymentStrategy deploymentStrategy = deployedArtifact.getRedeploymentStrategy();
            if (deploymentStrategy == DeploymentStrategy.STOP_START) {
                wlsTargets.add((WlsContainer) deployed.getContainer());
            }
        }
    }

    public static List<Step> startContainer(WlsContainer container, final int order) {
        switch (container.getDomain().getStartMode()) {
            case NodeManager:
                return ImmutableList.<Step>of(new StartWlsTargetStep(order, container));
            case Script:
                return Lists.transform(Lists.newArrayList(container.getServers()), new Function<Server, Step>() {
                    @Override
                    public Step apply(Server input) {
                        return new StartWlsServerWithScriptStep(order, input);
                    }
                });
            case WindowsService:
                throw new RuntimeException("The Windows Service mode has not been implemented yet");
        }
        throw new RuntimeException(String.format("Invalid startMode %s", container.getDomain().getStartMode()));
    }

    public static List<Step> stopContainer(WlsContainer container, int order) {
        return ImmutableList.<Step>of(new StopWlsTargetStep(order, container));
    }


}
