package com.xebialabs.deployit.plugin.wls.deployed;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.Maps;

import com.xebialabs.deployit.plugin.api.deployment.planning.Create;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.planning.Destroy;
import com.xebialabs.deployit.plugin.api.deployment.planning.Modify;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.DeployableArtifact;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.DerivedArtifact;
import com.xebialabs.deployit.plugin.api.validation.Placeholders;
import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.deployit.plugin.python.PythonManagedDeployed;
import com.xebialabs.deployit.plugin.wls.DeploymentStrategy;
import com.xebialabs.deployit.plugin.wls.container.StageMode;
import com.xebialabs.deployit.plugin.wls.container.WlsContainer;
import com.xebialabs.deployit.plugin.wls.contributor.ArtifactDeploymentContributor;
import com.xebialabs.deployit.plugin.wls.step.DeleteArtifactStep;
import com.xebialabs.deployit.plugin.wls.step.UploadArtifactStep;
import com.xebialabs.overthere.OverthereFile;

import static com.google.common.base.Strings.isNullOrEmpty;
import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.deployit.plugin.wls.CustomWlsOrders.UPLOAD_ARTIFACT;
import static com.xebialabs.deployit.plugin.wls.DeploymentStrategy.SIDE_BY_SIDE;
import static com.xebialabs.deployit.plugin.wls.DeploymentStrategy.STOP_START;

@SuppressWarnings("serial")
@Metadata(virtual = true, description = "Base class for all deployeds meant to contain Applications")
@Placeholders
public class ExtensibleDeployedArtifact<D extends DeployableArtifact> extends PythonManagedDeployed<D, WlsContainer> implements
        DerivedArtifact<DeployableArtifact> {

    static final int NO_RETIREMENT_TIMEOUT = -1;

    public static final Type DEPLOYED_SHARED_LIB_WAR = Type.valueOf("wls", "SharedLibraryWarModule");

    @Property(defaultValue = "Stage", required = false, description = "Indicates whether the artifact will be deployed as staged or nostage mode")
    private StageMode stageMode;

    @Property(required = false, description = "Absolute directory path where the artifact will be uploaded and used by the servers for nostage deployment mode")
    private String stagingDirectory;

    @Property(required = true, defaultValue = "CLASSIC", description = "Indicates what redeployment strategy to use for upgrading the application")
    private DeploymentStrategy redeploymentStrategy;

    @Property(required = false, description = "Indicates wither this artifact is to be deployed as a versioned application", category = "Versioning")
    private boolean versioned = false;

    @Property(required = false, defaultValue = "-1", description = "Timeout interval (in secs) before the previous application version is undeployed for side by side redeployment strategy", category = "Versioning")
    private int retireTimeout;

    @Property(required = false, category = "Versioning")
    private String versionIdentifier;

    @Property(required = false, defaultValue = "true", category = "Versioning", description = "If set, the 'versionIdentifier' property is managed by the plugin. Any value explicitly set for 'versionIdentifier' is ignored")
    private boolean automaticVersioning;

    @SuppressWarnings("unused")
    @Property(hidden = true, defaultValue = "id, name, type, deployable, properties, container, createScript, createVerb, createOrder, modifyScript, modifyVerb, modifyOrder, " +
            "destroyScript, destroyVerb, destroyOrder, startScript, startVerb, startOrder, stopScript, stopVerb, stopOrder, deploymentStrategy, placeholders, file, redeploymentStrategy, " +
            "securityPermissions, inheritPermissions, exposeDeployedApplication, stopRetiredApplicationOrder, undeployRetiredApplicationOrder, isRunningRetryWaitInterval, wlstPath, libraryScripts, discoverOrder, deploymentOrder ")
    private String standardPropertiesNotToExpose;

    @Property(category = "Placeholders", description = "A Map containing all the placeholders mapped to their values. Special values are &lt;ignore&gt; or &lt;empty&gt;", required = false)
    private Map<String, String> placeholders = newHashMap();

    @Property(hidden = true, required = false, description = "Python script invoked to deploy this Java EE artifact")
    private String createScript;

    @Property(hidden = true, defaultValue = "Deploy")
    private String createVerb;

    @Property(hidden = true, defaultValue = "70")
    private int createOrder;

    @Property(hidden = true, required = false, description = "Python script invoked to upgrade this Java EE artifact")
    private String modifyScript;

    @Property(hidden = true, defaultValue = "Upgrade")
    private String modifyVerb;

    @Property(hidden = true, defaultValue = "60")
    private int modifyOrder;

    @Property(hidden = true, required = false, description = "Python script invoked to undeploy this Java EE artifact")
    private String destroyScript;

    @Property(hidden = true, defaultValue = "Undeploy")
    private String destroyVerb;

    @Property(hidden = true, defaultValue = "30")
    private int destroyOrder;

    @Property(hidden = true, defaultValue = "98")
    private int undeployRetiredApplicationOrder;

    @Property(hidden = true, defaultValue = "95")
    private int stopRetiredApplicationOrder;

    @Property(hidden = true, required = false, description = "Python script invoked to start this Java EE artifact")
    private String startScript;

    @Property(hidden = true, defaultValue = "Start")
    private String startVerb;

    @Property(hidden = true, defaultValue = "90")
    private int startOrder;

    @Property(hidden = true, required = false, description = "Python script invoked to stop this Java EE artifact")
    private String stopScript;

    @Property(hidden = true, defaultValue = "Stop")
    private String stopVerb;

    @Property(hidden = true, defaultValue = "10")
    private int stopOrder;

    private OverthereFile derivedFile;

    private Map<String, Object> pythonVars = Maps.newHashMap();

    @Create
    public void deployArtifact(DeploymentPlanningContext result, Delta delta) {
        deployWeblogicArtifact(result, delta);
        // retireTimeout is only on "Modification"
        setRetireTimeout(NO_RETIREMENT_TIMEOUT);
    }

    void deployWeblogicArtifact(DeploymentPlanningContext result, Delta delta) {
        validateDeployedFields();
        handleCopyArtifact(result);
        addStep(result, createOrder, createScript, createVerb, checkpoint(delta, Operation.CREATE));
    }

    @SuppressWarnings({ "unchecked" })
    @Modify
    public void modifyArtifact(DeploymentPlanningContext result, Delta delta) {
        if (!isNullOrEmpty(modifyScript)) {
            addStep(result, modifyOrder, modifyScript, modifyVerb, checkpoint(delta));
        } else {
            ExtensibleDeployedArtifact<DeployableArtifact> previous = (ExtensibleDeployedArtifact<DeployableArtifact>) delta.getPrevious();
            ExtensibleDeployedArtifact<DeployableArtifact> current = (ExtensibleDeployedArtifact<DeployableArtifact>) delta.getDeployed();
            validateModifiedDeployed(previous, current);

            previous.destroyArtifact(result, delta);
            current.deployWeblogicArtifact(result, delta);
        }

        if (getRedeploymentStrategy() == SIDE_BY_SIDE) {
            Integer retireTimeout = getRetireTimeout();
            ArtifactDeploymentContributor.setWaitStepInterval(result, retireTimeout);
        } else {
            setProperty("retireTimeout", NO_RETIREMENT_TIMEOUT);
        }
    }

    @SuppressWarnings("unchecked")
    @Destroy
    public void destroyArtifact(DeploymentPlanningContext result, Delta delta) {
        int order = destroyOrder;
        if (delta.getOperation() == Operation.MODIFY
                && ((ExtensibleDeployedArtifact<DeployableArtifact>) delta.getDeployed()).getRedeploymentStrategy() == SIDE_BY_SIDE) {
            order = undeployRetiredApplicationOrder;
        }

        addStep(result, order, destroyScript, destroyVerb, false, checkpoint(delta, Operation.DESTROY));
        handleDeleteArtifact(result, destroyOrder);
    }

    private boolean isStartOrStopOperationRequired() {
        if (DEPLOYED_SHARED_LIB_WAR.equals(getType())) {
            return false;
        }

        if (getRedeploymentStrategy() == STOP_START) {
            return false;
        }

        return true;
    }

    @Create
    @Modify
    public void startApplication(DeploymentPlanningContext result) {
        if (isStartOrStopOperationRequired()) {
            addStep(result, startOrder, startScript, startVerb, false);
        }
    }

    @SuppressWarnings("rawtypes")
    @Modify
    public void stopApplicationForModifedArtifact(DeploymentPlanningContext result, Delta delta) {
        if (!isStartOrStopOperationRequired())
            return;

        final ExtensibleDeployedArtifact previous = (ExtensibleDeployedArtifact) delta.getPrevious();
        if (getRedeploymentStrategy() == DeploymentStrategy.SIDE_BY_SIDE) {
            previous.addStep(result, stopRetiredApplicationOrder, stopScript, stopVerb, false);
        } else {
            previous.addStep(result, stopOrder, stopScript, stopVerb, false);
        }
    }

    @Destroy
    public void stopApplication(DeploymentPlanningContext result) {
        if (isStartOrStopOperationRequired()) {
            addStep(result, stopOrder, stopScript, stopVerb, false);
        }
    }

    @Override
    protected String getDescription(String verb) {
        if (Artifact.class.isAssignableFrom(getDeployable().getClass())) {
            return String.format("%s %s on %s", verb, (getDeployable()).getFile().getName(), getContainer().getName());
        }
        return super.getDescription(verb);
    }

    @Override
    protected Map<String, Object> getPythonVars(DeployedApplication deployedApplication) {
        pythonVars.putAll(super.getPythonVars(deployedApplication));
        return pythonVars;

    }

    private void validateModifiedDeployed(ExtensibleDeployedArtifact<DeployableArtifact> previous, ExtensibleDeployedArtifact<DeployableArtifact> current) {
        validateDeployedFields();
        if (isNoStageMode()) {
            String previousStagingDir = previous.getStagingDirectory();
            String currentStagingDir = current.getStagingDirectory();
            if (previousStagingDir.equals(currentStagingDir)) {
                throw new RuntimeException("Can't deploy a new version to " + currentStagingDir
                        + ". This is same as the previous deployment nostage directory!");
            }
        }
    }

    private void validateDeployedFields() {
        if (isNoStageMode()) {
            if (isNullOrEmpty(getStagingDirectory())) {
                throw new RuntimeException("Staging directory shouldn't be blank");
            }

            if (getContainer().getHosts() == null) {
                throw new RuntimeException("NoStage deployment: the target " + getContainer() + " contains no hosts, check your repository");
            }

        }
        if (getRedeploymentStrategy() == SIDE_BY_SIDE && !isVersioned()) {
            throw new RuntimeException("SIDE_BY_SIDE deployment strategy is only applicable for versioned deployment");
        }
    }

    protected void handleCopyArtifact(DeploymentPlanningContext result) {
        // The noStage mode implies to manually copy the artifacts on all the hosts of the WlsTarget And the Host adminServer.
        if (isNoStageMode()) {
            final Set<Host> hosts = new HashSet<Host>(getContainer().getHosts());
            // Include always the host where is running the AdminServer
            hosts.add(getContainer().getDomain().getHost());

            String remoteDestinationFilename = getRemoteDestinationFilename(hosts.iterator().next());
            for (Host h : hosts) {
                ArtifactDeploymentContributor.addUploadArtifactStep(result, new UploadArtifactStep(UPLOAD_ARTIFACT, h, getDeployable(),
                        remoteDestinationFilename));
            }
            pythonVars.put("remoteDestinationFilename", remoteDestinationFilename);
        }
    }

    protected void handleDeleteArtifact(DeploymentPlanningContext result, int order) {
        // The noStage mode implies to manually copy the artifacts on all the hosts of the WlsTarget And the Host adminServer.
        if (isNoStageMode()) {
            final Set<Host> hosts = new HashSet<Host>(getContainer().getHosts());
            // Include always the host where is running the AdminServer
            hosts.add(getContainer().getDomain().getHost());
            for (Host h : hosts) {
                ArtifactDeploymentContributor.addDeleteArtifactStep(result, new DeleteArtifactStep(order, h, getRemoteDestinationFilename(h)));
            }
        }
    }

    private String getRemoteDestinationFilename(Host host) {
        return getStagingDirectory() + host.getOs().getFileSeparator() + getDeployable().getFile().getName();
    }

    private boolean isNoStageMode() {
        return getStageMode().equals(StageMode.NoStage);
    }

    public StageMode getStageMode() {
        return stageMode;
    }

    public void setStageMode(StageMode stageMode) {
        this.stageMode = stageMode;
    }

    public String getStagingDirectory() {
        return stagingDirectory;
    }

    public void setStagingDirectory(String stagingDirectory) {
        this.stagingDirectory = stagingDirectory;
    }

    public String getVersionIdentifier() {
        return versionIdentifier;
    }

    public void setVersionIdentifier(String versionIdentifier) {
        this.versionIdentifier = versionIdentifier;
    }

    public boolean isVersioned() {
        return versioned;
    }

    public void setVersioned(boolean versioned) {
        this.versioned = versioned;
    }

    public boolean isAutomaticVersioning() {
        return automaticVersioning;
    }

    public void setAutomaticVersioning(boolean automaticVersioning) {
        this.automaticVersioning = automaticVersioning;
    }

    public DeploymentStrategy getRedeploymentStrategy() {
        return redeploymentStrategy;
    }

    public int getRetireTimeout() {
        return retireTimeout;
    }

    public void setRetireTimeout(int retireTimeout) {
        this.retireTimeout = retireTimeout;
    }

    public void setRedeploymentStrategy(DeploymentStrategy deploymentStrategy) {
        this.redeploymentStrategy = deploymentStrategy;
    }

    @Override
    public OverthereFile getFile() {
        return derivedFile;
    }

    @Override
    public void setFile(OverthereFile file) {
        this.derivedFile = file;
    }

    @Override
    public DeployableArtifact getSourceArtifact() {
        return getDeployable();
    }

    @Override
    public Map<String, String> getPlaceholders() {
        return placeholders;
    }

    @Override
    public void setPlaceholders(Map<String, String> placeholders) {
        this.placeholders = placeholders;
    }
}
