package com.xebialabs.deployit.plugin.wls.contributor;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import com.xebialabs.deployit.plugin.api.deployment.planning.Contributor;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.udm.DeployableArtifact;
import com.xebialabs.deployit.plugin.wls.deployed.ExtensibleDeployedArtifact;
import com.xebialabs.deployit.plugin.wls.step.MultiTargetDeploymentStep;
import com.xebialabs.deployit.plugin.wls.step.MultiTargetDeploymentStep.StepData;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collection;
import java.util.Set;

import static com.google.common.collect.Sets.newHashSet;

public class MultiTargetArtifactDeploymentContributor {

    private static final String SINGLE_TARGET_DEPLOYMENT_STEPS = "singleTargetDeploymentSteps";

    @Contributor
    public static void contributeMultiTargetDeploymentSteps(@SuppressWarnings("unused") Deltas allDeltas, DeploymentPlanningContext ctx) {
        Multimap<StepGroupKey, StepData> groupedSteps = groupStepsByOperationAndArtifact(ctx);

        for (StepGroupKey key : groupedSteps.keySet()) {
            Collection<StepData> singleSteps = groupedSteps.get(key);

            if (isCreateModifyOrDestroyStep(key)) {
                MultiTargetDeploymentStep step = createMultiTargetDeploymentStep(ctx, singleSteps);
                if (step.getDeltas().isEmpty()) {
                    ctx.addStep(step);
                } else {
                    ctx.addStepWithCheckpoint(step, step.getDeltas());
                }
            } else if (isStartStopStep(key)) {
                MultiTargetDeploymentStep step = createMultiTargetDeploymentStep(ctx, singleSteps);
                ctx.addStep(step);
            } else {
                logger.warn("ignore steps not applicable for multi-target deployment {}", singleSteps);
            }
        }
    }

    protected static Multimap<StepGroupKey, StepData> groupStepsByOperationAndArtifact(DeploymentPlanningContext ctx) {
        Multimap<StepGroupKey, StepData> stepsGroupedByOpOnArtifact = HashMultimap.create();
        for (StepData target : getDeploymentSteps(ctx)) {
            stepsGroupedByOpOnArtifact.put(new StepGroupKey(target.getVerb(), target.getOrder(), target.getDeployedArtifact()), target);
        }
        logger.trace("steps grouped for multi-target deployment {}", stepsGroupedByOpOnArtifact);
        return stepsGroupedByOpOnArtifact;
    }

    protected static boolean isCreateModifyOrDestroyStep(StepGroupKey key) {
        return key.verb.equals(key.deployed.getCreateVerb()) || key.verb.equals(key.deployed.getModifyVerb()) || key.verb.equals(key.deployed.getDestroyVerb());
    }

    protected static boolean isStartStopStep(StepGroupKey key) {
        return key.verb.equals(key.deployed.getStartVerb()) || key.verb.equals(key.deployed.getStopVerb());
    }

    protected static MultiTargetDeploymentStep createMultiTargetDeploymentStep(DeploymentPlanningContext ctx, Collection<StepData> singleTargeSteps) {
        MultiTargetDeploymentStep multiTargetStep = MultiTargetDeploymentStep.create(ctx, singleTargeSteps);
        logger.debug("add multi-target step to plan --> {}", multiTargetStep.getDescription());
        return multiTargetStep;
    }

    public static void storeDeploymentStep(DeploymentPlanningContext ctx, ExtensibleDeployedArtifact<? extends DeployableArtifact> deployedArtifact, int order, String scriptPath, String verb, Set<String> options, Delta delta) {
        StepData target = new StepData(deployedArtifact, order, scriptPath, verb, options, delta);
        getDeploymentSteps(ctx).add(target);
    }

    @SuppressWarnings("unchecked")
    private static Set<StepData> getDeploymentSteps(DeploymentPlanningContext ctx) {
        Set<StepData> targets = (Set<StepData>) ctx.getAttribute(SINGLE_TARGET_DEPLOYMENT_STEPS);
        if (targets == null) {
            targets = newHashSet();
            ctx.setAttribute(SINGLE_TARGET_DEPLOYMENT_STEPS, targets);
        }
        return targets;
    }

    protected static final Logger logger = LoggerFactory.getLogger(MultiTargetArtifactDeploymentContributor.class);

    protected static class StepGroupKey {
        public final String verb;
        public final int order;
        public ExtensibleDeployedArtifact<?> deployed;

        public StepGroupKey(String verb, int order, ExtensibleDeployedArtifact<?> deployed) {
            this.verb = verb;
            this.order = order;
            this.deployed = deployed;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            StepGroupKey that = (StepGroupKey) o;

            if (order != that.order) return false;
            if (!deployed.getName().equals(that.deployed.getName())) return false;
            if (!deployed.getContainer().getDomain().getName().equals(that.deployed.getContainer().getDomain().getName()))
                return false;
            if (!verb.equals(that.verb)) return false;

            return true;
        }

        @Override
        public int hashCode() {
            int result = verb.hashCode();
            result = 31 * result + order;
            result = 31 * result + deployed.getName().hashCode();
            result = 31 * result + deployed.getContainer().getDomain().getName().hashCode();
            return result;
        }

        @Override
        public String toString() {
            final StringBuffer sb = new StringBuffer("Key{");
            sb.append("verb='").append(verb).append('\'');
            sb.append(", order=").append(order);
            sb.append(", name=").append(deployed.getName());
            sb.append('}');
            return sb.toString();
        }
    }
}
