package com.xebialabs.deployit.plugin.wls.preplanprocessor;

import java.io.*;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.HashMap;
import java.util.Map;
import java.util.jar.Attributes;
import java.util.jar.Manifest;

import com.xebialabs.overthere.OverthereFile;
import de.schlichtherle.truezip.file.TVFS;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.ImmutableMap;

import com.xebialabs.deployit.plugin.api.deployment.planning.PrePlanProcessor;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.wls.deployed.ExtensibleDeployedArtifact;
import com.xebialabs.deployit.plugin.wls.freemarker.TemplateEvaluator;
import com.xebialabs.deployit.plugin.wls.utils.Closeables;
import com.xebialabs.overthere.RuntimeIOException;

import de.schlichtherle.truezip.file.TFile;
import de.schlichtherle.truezip.file.TFileInputStream;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Strings.isNullOrEmpty;
import static com.google.common.collect.Iterables.filter;
import static com.google.common.collect.Iterables.transform;

public class WlsArtifactProcessor {

    private static final Logger logger = LoggerFactory.getLogger(WlsArtifactProcessor.class);

    @PrePlanProcessor
    public static Step setVersionIdentifiers(DeltaSpecification spec) {

        final String appVersion = spec.getDeployedApplication().getVersion().getVersion();
        final String appName = spec.getDeployedApplication().getVersion().getApplication().getName();
        final String defaultVersion = appName + "-" + appVersion;

        final VersionSetter version = new VersionSetter(defaultVersion);
        for (ExtensibleDeployedArtifact<?> artifact : filter(transform(spec.getDeltas(), TO_DEPLOYED), IS_VERSIONED)) {
            if (isNullOrEmpty(artifact.getVersionIdentifier()) || artifact.isAutomaticVersioning()) {
                version.apply(artifact);
            }
        }
        return null;
    }

    private static final Function<Delta, ExtensibleDeployedArtifact<?>> TO_DEPLOYED = delta -> {
        Deployed<?, ?> deployed = delta.getDeployed();
        return deployed instanceof ExtensibleDeployedArtifact ? (ExtensibleDeployedArtifact<?>) deployed : null;
    };

    private static final Predicate<ExtensibleDeployedArtifact<?>> IS_VERSIONED = input -> Predicates.notNull().apply(input) && input.isVersioned();


    private static class VersionSetter implements Function<ExtensibleDeployedArtifact<?>, Void> {

        private final String defaultVersion;

        public VersionSetter(String defaultVersion) {
            this.defaultVersion = defaultVersion;
        }

        @Override
        public Void apply(ExtensibleDeployedArtifact<?> deployed) {
            final Artifact artifact = deployed.getDeployable();
            final Manifest manifest = readManifest(artifact);
            if (manifest == null) {
                logger.debug("Unable to read manifest for {}", artifact.getFile().getName());
                return null;
            }
            final Attributes mainAttributes = manifest.getMainAttributes();
            final String extName = mainAttributes.getValue(deployed.<String>getProperty("manifestVersionProperty"));

            deployed.setVersioned(true);
            if (isNullOrEmpty(extName)) {
                logger.debug("Setting version identifier to default value '{}'", defaultVersion);
                deployed.setVersionIdentifier(defaultVersion);
            } else {
                Map<String,Object> attributesAsMap =  new HashMap<>();
                for (Object key : mainAttributes.keySet()) {
                    attributesAsMap.put(key.toString(), mainAttributes.getValue(key.toString()));
                }
                final ImmutableMap<String, Object> vars = ImmutableMap.<String, Object>of("manifestAttributes", attributesAsMap);
                String versionIdentifier = TemplateEvaluator.evaluateExpression(deployed.<String>getProperty("versionExpression"), vars);
                versionIdentifier = versionIdentifier.replace('\n',' ').replace("\r\n"," ").trim();
                logger.debug("Setting version identifier to manifest-derived values '{}'", versionIdentifier);
                deployed.setVersionIdentifier(versionIdentifier);
            }
            return null;
        }

        private Manifest readManifest(Artifact artifact) {
            InputStream is = null;
            OverthereFile file = artifact.getFile();
            String fileName = file.getName();
            Path tmpPath = null;
            TFile manifestTFile = null;
            try {
                tmpPath = Files.createTempFile("manifest", fileName);
                Files.copy(file.getInputStream(), tmpPath, StandardCopyOption.REPLACE_EXISTING);
                manifestTFile = new TFile(new TFile(tmpPath.toFile()), "META-INF/MANIFEST.MF");
                checkArgument(manifestTFile.exists(), "The archive [%s] does not contain a MANIFEST.MF file.", fileName);
                is = new TFileInputStream(manifestTFile);
                return new Manifest(is);
            } catch (FileNotFoundException fnfe) {
                throw new RuntimeIOException(String.format("The archive [%s] does not contain a MANIFEST.MF file.", fileName), fnfe);
            } catch (IOException e) {
                throw new RuntimeIOException(String.format("Error reading the MANIFEST.MF in archive [%s].", fileName), e);
            } finally {
                Closeables.closeQuietly(is);
                ignoreException(tmpPath, Files::delete);
                ignoreException(manifestTFile, TVFS::umount);
            }
        }
    }

    private static <T> void ignoreException(T value, ConsumerWithException<T> consumer) {
        try {
            if(value != null) {
                consumer.consume(value);
            }
        } catch (Exception ignored) {
        }
    }

    interface ConsumerWithException<T> {
        void consume(T value) throws Exception;
    }
}
