package com.crossingchannels.portal.websphere.ci.portlet.model;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang.ObjectUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.crossingchannels.portal.websphere.model.AbstractModelElementFactory;
import com.crossingchannels.portal.websphere.specification.jee5.EarArchive;
import com.crossingchannels.portal.websphere.specification.jee5.EarDescriptor;
import com.crossingchannels.portal.websphere.specification.jee5.IllegalDescriptorFoundException;
import com.crossingchannels.portal.websphere.specification.jee5.WebArchive;
import com.xebialabs.deployit.plugin.was.deployed.ExtensibleDeployedArtifact;

import ext.deployit.com.crossingchannels.portal.websphere.ci.generic.step.DetermineWebArchiveInstallationRootStep;
import ext.deployit.com.crossingchannels.portal.websphere.ci.portlet.deployed.BasePortlet;

/**
 * Builder to create different types of model elements
 * 
 * @author FWiegerinck
 */
public class PortletArchiveBuilder extends AbstractModelElementFactory<PortletArchive> {
	
	private static final Logger LOGGER = LoggerFactory.getLogger(PortletArchiveBuilder.class);
	
    /**
     * Define pattern to update websphere variable to XML access variable.
     */
    private static final Pattern WEBSPHERE_VARIABLE_PATTERN = Pattern.compile("\\$\\((.*)\\)");

    /**
     * Define name of CI attribute 'contextRoot'. CI attribute is provided by wp.War
     */
    private static final String CI_ATTR_WAR_CONTEXTROOT = "contextRoot";

    /**
     * Define name of CI attribute 'autoScanForPortlets'. CI attribute is provided by portlet archives (wp.Ear, wp.War)
     */
	private static final String CI_ATTR_ARCHIVE_AUTOSCAN = "autoScanForPortlets";
    
    public static PortletArchiveBuilder fromEar(final ExtensibleDeployedArtifact artifactToInstall) {
        return new PortletArchiveBuilder(artifactToInstall).initializeAsEar();
    }

    public static PortletArchiveBuilder fromWar(final ExtensibleDeployedArtifact artifactToInstall) {
        return new PortletArchiveBuilder(artifactToInstall).initializeAsWar();
    }

    private final ExtensibleDeployedArtifact artifactToInstall;
    private final Map<String, String> webModulesMapping;
    private final Map<String, WebArchive> webArchives;

    /**
     * Hide constructor for utility/builder class
     */
    private PortletArchiveBuilder(final ExtensibleDeployedArtifact artifactToInstall) {
        super();

        // Store information
        this.artifactToInstall = artifactToInstall;
        this.webModulesMapping = new HashMap<String, String>();
        this.webArchives = new HashMap<String, WebArchive>();
    }

    private PortletArchiveBuilder initializeAsEar() {

        // Read EAR file
        try {
            final EarArchive earArchive = EarArchive.fromDisk(this.artifactToInstall.getDeployable().getFile().getPath());
            final EarDescriptor earDescriptor = earArchive.getDescriptor();
            for (final String webArchiveURI : earDescriptor.getWebArchiveURIs()) {
                this.webModulesMapping.put(webArchiveURI, earDescriptor.getWebArchiveContextRoot(webArchiveURI));
                this.webArchives.put(webArchiveURI, earArchive.getWebArchive(webArchiveURI));
            }
        } catch (final IOException e) {
            throw new RuntimeException("Cannot open EAR archive from repo", e);
        } catch (final IllegalDescriptorFoundException e) {
            throw new RuntimeException("Cannot open EAR descriptor from EAR in repo", e);
        }

        return this;
    }

    private PortletArchiveBuilder initializeAsWar() {

        // Get WAR info
        final String contextRoot = this.artifactToInstall.getProperty(PortletArchiveBuilder.CI_ATTR_WAR_CONTEXTROOT);
        final String archiveName = this.artifactToInstall.getDeployable().getFile().getName();
        
        PortletArchiveBuilder.LOGGER.trace("Use WAR filename [{}] for deployment of CI [{}]", archiveName, this.artifactToInstall.getId());
        
        this.webModulesMapping.put(archiveName, contextRoot);
        try {
            this.webArchives.put(archiveName, WebArchive.fromDisk(this.artifactToInstall.getDeployable().getFile().getPath()));
        } catch (final IOException e) {
            throw new RuntimeException("Cannot open WAR archive from repo", e);
        }

        return this;
    }

    protected String getApplicationInstallDirectory() {
    	if ( this.artifactToInstall.hasProperty(DetermineWebArchiveInstallationRootStep.PROPERTY_INSTALL_ROOT)) {
    		return this.artifactToInstall.getProperty(DetermineWebArchiveInstallationRootStep.PROPERTY_INSTALL_ROOT);
    	} else {
    		return null;
    	}
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public PortletArchive toModel() {
        // Retrieve all required information
        final String locationOnDisk = PortletArchiveBuilder.reformatVariables(this.getApplicationInstallDirectory());
        final List<WebModule> webModules = new ArrayList<WebModule>();

        // Get set of portlets
        boolean autoScanForPortlets = (Boolean) ObjectUtils.defaultIfNull( this.artifactToInstall.getProperty(CI_ATTR_ARCHIVE_AUTOSCAN), Boolean.FALSE);
        Set<BasePortlet<?, ?>> portlets = this.artifactToInstall.getProperty("portlets");

        for (final String webArchiveURI : this.webModulesMapping.keySet()) {
            final String webModuleLocationOnDisk;
            if ( locationOnDisk != null ) {
            	webModuleLocationOnDisk = locationOnDisk + "/" + webArchiveURI;
            } else {
            	webModuleLocationOnDisk = null;
            }
            try {
                webModules.add(WebModuleBuilder.create(webModuleLocationOnDisk, this.webModulesMapping.get(webArchiveURI)).use(this.webArchives.get(webArchiveURI), portlets, autoScanForPortlets).toModel());
            } catch (final IllegalDescriptorFoundException e) {
                throw new RuntimeException("Cannot process EAR archive from repo", e);

            } catch (final IOException e) {
                throw new RuntimeException("Cannot process EAR archive from repo", e);

            }
        }

        // Do generate items
        return new PortletArchive(locationOnDisk, webModules);
    }

    private static String reformatVariables(final String value) {
    	// Return direct when value is NULL
    	if ( value == null ) {
    		return null;
    	}
    	
        final Matcher m = PortletArchiveBuilder.WEBSPHERE_VARIABLE_PATTERN.matcher(value);
        final StringBuffer result = new StringBuffer();
        while (m.find()) {
            m.appendReplacement(result, "\\$" + m.group(1).toLowerCase() + "\\$");
        }
        m.appendTail(result);
        return result.toString();
    }
}
