package com.crossingchannels.portal.websphere.specification.jee5;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Set;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.Validate;

import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.local.LocalConnection;

/**
 * Represents an Enterprise Application Archive.
 * 
 * @author FWiegerinck
 */
public class EarArchive {

    /**
     * Load archive from disk
     * 
     * @param location
     *            Location of the archive.
     * @return Instance of EarArchive.
     * @throws IOException
     */
    public static EarArchive fromDisk(final String location) throws IOException {
        Validate.notNull(location);

        final File fileRef = new File(location);
        // Ensure file is valid
        if (!fileRef.exists() || !fileRef.canRead()) {
            throw new FileNotFoundException(String.format("Unable to read file [%s]", location));
        }

        return new EarArchive(fileRef);
    }

    private final File archiveOnDisk;

    private EarDescriptor earDescriptor = null;

    private final Set<String> webArchives;

    private EarArchive(final File archive) throws IOException {

        // Save information
        this.archiveOnDisk = archive;
        this.webArchives = new HashSet<String>();
        final JarFile earFile = new JarFile(archive);
        try {
            final Enumeration<JarEntry> jarEntries = earFile.entries();
            while (jarEntries.hasMoreElements()) {
                final JarEntry jarEntry = jarEntries.nextElement();
                if (jarEntry.getName().endsWith(".war")) {
                    this.webArchives.add(jarEntry.getName());
                }
            }
        } finally {
            earFile.close();
        }

    }

    public EarDescriptor getDescriptor() throws IllegalDescriptorFoundException, IOException {
        if (this.earDescriptor == null) {
            this.earDescriptor = this.loadDescriptor();
        }

        return this.earDescriptor;
    }

    public WebArchive getWebArchive(final String webArchiveURI) throws IOException {
        if (this.webArchives.contains(webArchiveURI)) {

            final JarFile earFile = new JarFile(this.archiveOnDisk);
            try {
                final InputStream stream = earFile.getInputStream(earFile.getEntry(webArchiveURI));
                final String temporaryFileName = this.copyToTemporaryFile(stream, webArchiveURI);
                return WebArchive.fromDisk(temporaryFileName);
            } finally {
                earFile.close();
            }
            
        } else {
            throw new RuntimeException("illegal web archive requested. URI is unknown");
        }
    }

    /**
     * Copy the content of the stream to a temporary file with a filename with the specified prefix.
     * 
     * @param stream
     *            The inputstream containing the content of the WAR file.
     * @param prefix
     *            The name prefix used to create the temporary file.
     * @return Location of the tempoary file.
     * @throws IOException
     *             Thrown if the content of the WAR could not be written to a temporary file.
     */
    private String copyToTemporaryFile(final InputStream stream, final String prefix) throws IOException {
        final OverthereFile tempFile = LocalConnection.getLocalConnection().getTempFile(prefix, ".war");
        OutputStream outputStream = tempFile.getOutputStream();
        try {
            IOUtils.copy(stream, outputStream);
        } finally {
        	IOUtils.closeQuietly(outputStream);
        }
        return tempFile.getPath();
    }

    /**
     * @return The EarDescriptor from the archive
     * @throws IllegalDescriptorFoundException
     * @throws IOException
     */
    private EarDescriptor loadDescriptor() throws IllegalDescriptorFoundException, IOException {
        final JarFile earFile = new JarFile(this.archiveOnDisk);
        final InputStream descriptor = earFile.getInputStream(earFile.getJarEntry("META-INF/application.xml"));
        try {
            return EarDescriptor.fromInputStream(descriptor);
        } finally {
            earFile.close();
        }
    }
}
