package com.crossingchannels.portal.websphere.specification.jee5;

import java.io.InputStream;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;

import com.crossingchannels.portal.websphere.specification.jee5.dom.DisplayNameType;
import com.crossingchannels.portal.websphere.specification.jee5.dom.ObjectFactory;
import com.crossingchannels.portal.websphere.specification.jee5.dom.WebAppType;
import com.crossingchannels.portal.websphere.util.ChangeNamespaceXmlFilter;
import com.crossingchannels.portal.websphere.util.UnexpectedXmlElementException;
import com.crossingchannels.portal.websphere.util.XmlUtil;

/**
 * Represents an JEE version 5 compatible WAR descriptor.
 * 
 * @author FWiegerinck
 */
public class WebDescriptor {

    private static final Set<String> COMPATIBLE_NAMESPACES = Collections.unmodifiableSet(new HashSet<String>(Arrays.asList("http://java.sun.com/xml/ns/javaee", "http://java.sun.com/xml/ns/j2ee")));

    private static final String TARGET_NAMESPACE = "http://java.sun.com/xml/ns/javaee";	
	
    /**
     * Create a new WebDescriptor object using the supplied XML file.
     * 
     * @param webXmlInputStream
     *            InputStream containing the XML file.
     * @return Instance of WebDescriptor.
     * @throws IllegalDescriptorFoundException
     *             Thrown if the descriptor is not valid.
     */
    public static final WebDescriptor fromInputStream(final InputStream webXmlInputStream) throws IllegalDescriptorFoundException {
        Validate.notNull(webXmlInputStream);
        // Load DOM
        final WebAppType domApplication;
        try {
            domApplication = XmlUtil.loadDom(webXmlInputStream, ObjectFactory.class.getPackage(), WebAppType.class, new ChangeNamespaceXmlFilter(WebDescriptor.TARGET_NAMESPACE,
                    WebDescriptor.COMPATIBLE_NAMESPACES));
        } catch (final JAXBException e) {
            throw new IllegalDescriptorFoundException("WAR descriptor contains invalid XML", e);
        } catch (final UnexpectedXmlElementException e) {
            throw new IllegalDescriptorFoundException("WAR descriptor contains illegal XML root element", e);
        }

        // Create return object
        final WebDescriptor webDescriptor = new WebDescriptor();
        webDescriptor.setUniqueIdentifier(domApplication.getId());
        for (final JAXBElement<?> element : domApplication.getDescriptionAndDisplayNameAndIcon()) {
            final Object value = element.getValue();
            if (value instanceof DisplayNameType) {
                final DisplayNameType displayNameType = DisplayNameType.class.cast(value);

                // Skip non default language display name
                if (StringUtils.isEmpty(displayNameType.getLang())) {
                    webDescriptor.setDisplayName(displayNameType.getValue());
                }
            }
        }

        return webDescriptor;
    }

    private String displayName;
    private String uniqueIdentifier;

    /**
     * @return the displayName
     */
    public String getDisplayName() {
        return this.displayName;
    }

    /**
     * @param displayName
     *            the displayName to set
     */
    public void setDisplayName(final String displayName) {
        this.displayName = displayName;
    }

    /**
     * @return the uniqueIdentifier
     */
    public String getUniqueIdentifier() {
        return this.uniqueIdentifier;
    }

    /**
     * @param uniqueIdentifier
     *            the uniqueIdentifier to set
     */
    public void setUniqueIdentifier(final String uniqueIdentifier) {
        this.uniqueIdentifier = uniqueIdentifier;
    }

}
