package com.crossingchannels.portal.websphere.specification.jsr286;

import java.io.InputStream;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import javax.xml.bind.JAXBException;

import org.apache.commons.lang.Validate;

import com.crossingchannels.portal.websphere.specification.jee5.IllegalDescriptorFoundException;
import com.crossingchannels.portal.websphere.specification.jsr286.dom.ObjectFactory;
import com.crossingchannels.portal.websphere.specification.jsr286.dom.PortletAppType;
import com.crossingchannels.portal.websphere.specification.jsr286.dom.PortletType;
import com.crossingchannels.portal.websphere.util.ChangeNamespaceXmlFilter;
import com.crossingchannels.portal.websphere.util.UnexpectedXmlElementException;
import com.crossingchannels.portal.websphere.util.XmlUtil;

/**
 * Represents an JSR286 compatible portlet descriptor.
 * 
 * @author FWiegerinck
 */
public class PortletDescriptor {

    private static final Set<String> COMPATIBLE_NAMESPACES = Collections.unmodifiableSet(new HashSet<String>(Arrays.asList("http://java.sun.com/xml/ns/portlet/portlet-app_1_0.xsd",
            "http://java.sun.com/xml/ns/portlet/portlet-app_2_0.xsd")));

    private static final String TARGET_NAMESPACE = "http://java.sun.com/xml/ns/portlet/portlet-app_2_0.xsd";

    /**
     * Create a new PortletDescriptor object using the supplied XML file.
     * 
     * @param portletXmlInputStream
     *            InputStream containing the XML file.
     * @return Instance of PortletDescriptor.
     * @throws IllegalDescriptorFoundException
     *             Thrown if the descriptor is not valid.
     */
    public static final PortletDescriptor fromInputStream(final InputStream portletXmlInputStream) throws IllegalDescriptorFoundException {
        Validate.notNull(portletXmlInputStream);
        // Load DOM
        final PortletAppType domApplication;
        try {
            domApplication = XmlUtil.loadDom(portletXmlInputStream, ObjectFactory.class.getPackage(), PortletAppType.class, new ChangeNamespaceXmlFilter(PortletDescriptor.TARGET_NAMESPACE,
                    PortletDescriptor.COMPATIBLE_NAMESPACES));
        } catch (final JAXBException e) {
            throw new IllegalDescriptorFoundException("Portlet descriptor contains invalid XML", e);
        } catch (final UnexpectedXmlElementException e) {
            throw new IllegalDescriptorFoundException("Portlet descriptor contains illegal XML root element", e);
        }

        // Create return object
        final PortletDescriptor portletDescriptor = new PortletDescriptor();
        portletDescriptor.setUniqueIdentifier(domApplication.getId());
        portletDescriptor.setName(domApplication.getId());

        final Set<String> portletNames = new HashSet<String>();
        for (final PortletType domPortlet : domApplication.getPortlet()) {
            portletNames.add(domPortlet.getPortletName().getValue().trim());
        }
        portletDescriptor.setPortletNames(portletNames);

        return portletDescriptor;
    }

    private String name;
    private String uniqueIdentifier;
    private Set<String> portletNames;

    /**
     * @return the name
     */
    public String getName() {
        return this.name;
    }

    /**
     * @return the uniqueIdentifier
     */
    public String getUniqueIdentifier() {
        return this.uniqueIdentifier;
    }

    /**
     * @return the portletNames
     */
    public Set<String> getPortletNames() {
        return this.portletNames;
    }

    /**
     * @param name
     *            the name to set
     */
    private void setName(final String name) {
        this.name = name;
    }

    /**
     * @param uniqueIdentifier
     *            the uniqueIdentifier to set
     */
    private void setUniqueIdentifier(final String uniqueIdentifier) {
        this.uniqueIdentifier = uniqueIdentifier;
    }

    /**
     * @param portletNames
     *            the portletNames to set
     */
    private void setPortletNames(final Set<String> portletNames) {
        this.portletNames = Collections.unmodifiableSet(portletNames);
    }

}
