package com.xebialabs.deployit.plugin.cloud.step;


import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import com.google.common.base.Predicate;
import com.google.common.collect.Maps;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.cloud.ci.BaseHostTemplate;
import com.xebialabs.deployit.plugin.cloud.util.CiParser;
import com.xebialabs.deployit.plugin.cloud.util.ContextHelper;
import com.xebialabs.deployit.plugin.cloud.util.InstanceDescriptorResolver;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Collections2.filter;

/**
 * Registers created hosts in repository.
 * This step depends on context attributes {@code ContextAttribute.CREATED_INSTANCES} and {@code ContextAttribute.USED_TEMPLATES}, which
 * should contain created instances and used templates in the proper order.
 * This step always should be executed after some step, which actually triggers instantiation on cloud platform and writes
 * appropriate objects to the context.
 */
@SuppressWarnings("serial")
public class RegisterInstancesStep implements Step {

    /**
     * Context attribute key which corresponds to the collection of cloud hosts stored in repository
     */
    public final static String HOSTS_MAP_CTX_ATTRIBUTE = "hosts";

    /**
     * Directory which should contain created hosts
     */
    private final String hostsPath;

    private InstanceDescriptorResolver descriptorResolver;

    private CiParser ciParser;

    public RegisterInstancesStep(final InstanceDescriptorResolver descriptorResolver, CiParser ciParser, String hostsPath) {
        this.descriptorResolver = descriptorResolver;
        this.ciParser = ciParser;
        this.hostsPath = hostsPath;
    }

    @Override
    public int getOrder() {
        return DEFAULT_ORDER + 30;
    }

    @Override
    public String getDescription() {
        return "Register created instances in repository";
    }

    @Override
    @SuppressWarnings("unchecked")
    public StepExitCode execute(final ExecutionContext ctx) throws Exception {

        List<String> instances;
        List<BaseHostTemplate> templates;

        try {
            instances = checkNotNull((List<String>) ctx.getAttribute(ContextAttribute.CREATED_INSTANCES.name()));
            templates = checkNotNull((List<BaseHostTemplate>) ctx.getAttribute(ContextAttribute.USED_TEMPLATES.name()));
        } catch (NullPointerException e) {
            ctx.logError("Can not find required context attribute", e);
            return StepExitCode.FAIL;
        }

        for(int i=0; i < instances.size(); i++) {

            BaseHostTemplate instanceTpl = templates.get(i);
            String cloudId = instances.get(i);
            String instancePublicAddress = instanceTpl.getInstanceIpAddress(cloudId);

            HashMap<String, Object> scope = Maps.newHashMap();
            scope.put("hostsPath", hostsPath);
            scope.put("hostTemplate", instanceTpl);
            scope.put("cloudId", cloudId);
            scope.put("hostAddress", instancePublicAddress);

            String descriptor = descriptorResolver.resolve(instanceTpl.getXmlDescriptor(), scope);

            List<ConfigurationItem> cis = ciParser.fromString(descriptor, ctx.getRepository());

            for (ConfigurationItem ci : cis) {
                ctx.logOutput("Configuration item " + ci.getId() + " has been created");
            }

            ctx.getRepository().create(cis.toArray(new ConfigurationItem[cis.size()]));

            ctx.logOutput("Host " + instancePublicAddress + " with underlying middleware is created");

            Collection<ConfigurationItem> hosts = filter(cis, new Predicate<ConfigurationItem>() {
                @Override
                public boolean apply(final ConfigurationItem ci) {
                    return ci.getType().getDescriptor().isAssignableTo(Type.valueOf("overthere.Host"));
                }
            });

            ContextHelper.wrapped(ctx).safeSet(HOSTS_MAP_CTX_ATTRIBUTE, new HashSet<ConfigurationItem>(), hosts);
        }


        return StepExitCode.SUCCESS;
    }

}
