package com.xebialabs.deployit.plugin.cloud.util;

import java.io.ByteArrayInputStream;
import java.io.Serializable;
import java.util.List;
import java.util.Set;
import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableSet;
import com.thoughtworks.xstream.io.StreamException;
import com.thoughtworks.xstream.io.xml.StaxDriver;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.services.Repository;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.xltype.serialization.CiReference;
import com.xebialabs.xltype.serialization.ConfigurationItemConverter;
import com.xebialabs.xltype.serialization.xstream.CiXstreamReader;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Sets.difference;
import static com.google.common.collect.Sets.newHashSet;

@SuppressWarnings("serial")
public class CiParser implements Serializable {

    /**
     * Converts xml string into the list of CIs.
     * @param source XML string
     * @param repository Repository for reference resolution
     * @return parsed CIs
     */
    public List<ConfigurationItem> fromString(String source, Repository repository) {
        return fromString(source, repository, ImmutableSet.<String>of());
    }

    /**
     * Converts xml string into the list of CIs.
     * @param source XML string
     * @param repository Repository for reference resolution
     * @param externalReferences References to be considered as resolvable
     * @return parsed CIs
     */
    public List<ConfigurationItem> fromString(String source, Repository repository, final Set<String> externalReferences) {
        try {
            CiXstreamReader reader = new CiXstreamReader(new StaxDriver().createReader(new ByteArrayInputStream(source.getBytes())));
            ConfigurationItemConverter converter = new ConfigurationItemConverter();
            List<ConfigurationItem> configurationItems = converter.readCis(reader);

            resolveReferencesSilently(repository, converter);

            for (CiReference r : converter.getReferences()) {
                if (difference(newHashSet(r.getIds()), externalReferences).isEmpty()) {
                    continue;
                }

                PropertyDescriptor pd = r.getProperty();
                switch (pd.getKind()) {
                    case LIST_OF_CI:
                    case SET_OF_CI:
                        @SuppressWarnings("unchecked")
                        List<ConfigurationItem> cis = newArrayList((Iterable<ConfigurationItem>) pd.get(r.getCi()));

                        if(cis.size() < r.getIds().size()) {
                            throw new ReferenceNotFoundException(r);
                        }

                        for (ConfigurationItem ci : cis) {
                            if(ci == null) {
                                throw new ReferenceNotFoundException(r);
                            }
                        }

                        break;
                    default:
                        if(pd.get(r.getCi()) == null) {
                            throw new ReferenceNotFoundException(r);
                        }
                }
            }


            return configurationItems;
        } catch (StreamException e) {
            // Erasing column information because spaces are not shown correctly on the UI.
            throw new StreamException(e.getMessage().replaceAll("\\[row,col\\]\\:\\[([0-9]+),([0-9]+)\\]", "row: $1"));
        }
    }

    private void resolveReferencesSilently(Repository repository, ConfigurationItemConverter ciConverter) {
        for (CiReference reference : ciConverter.getReferences()) {

            // Resolve IDs to CIs
            List<ConfigurationItem> resolvedCIs = newArrayList();
            for (String id : reference.getIds()) {
                ConfigurationItem alsoRead = ciConverter.getReadCIs().get(id);
                if (alsoRead != null) {
                    resolvedCIs.add(alsoRead);
                } else if(repository.exists(id)) {
                    resolvedCIs.add(repository.read(id));
                }
            }

            // Set referred CIs on the parsed CI
            reference.set(resolvedCIs);
        }
    }

    public class ReferenceNotFoundException extends RuntimeException {
        public ReferenceNotFoundException(CiReference r) {
            super("Can not find reference from {" + r.getCi().getId() + "}." + r.getProperty().getName() + " to " + Joiner.on(", ").join(r.getIds()));
        }
    }

}
