package com.xebialabs.deployit.plugin.cloud.step;

import java.util.List;

import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Container;
import com.xebialabs.deployit.plugin.cloud.ci.Environment;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Lists.newArrayList;

@SuppressWarnings("serial")
public class CheckEnvironmentStep implements Step {

    private static final String ENVIRONMENTS_ROOT = "Environments";

    private String environmentId;

    public CheckEnvironmentStep(final String environmentId) {
        this.environmentId = environmentId;
    }

    @Override
    public String getDescription() {
        return "Check if containers from " + environmentId + " are used in other environments and other way around";
    }

    @Override
    public int getOrder() {
        return DEFAULT_ORDER;
    }

    @Override
    public StepExitCode execute(final ExecutionContext ctx) throws Exception {

        Environment targetEnv = (Environment)checkNotNull(ctx.getRepository().read(environmentId));

        boolean errors = false;

        // Check if external containers are added to the environment
        for (final Container container : targetEnv.getMembers()) {
            if (!belongsToOneOfTopLevelContainers(container, targetEnv)) {
                ctx.logOutput("Container " + container.getId() + " is used as a member of environment " + environmentId + " but was not created with it.");
                ctx.logOutput("Please remove the reference before destroying " + environmentId);
                errors = true;
            }
        }


        // Check if containers from this dynamic env are used in other envs
        List<com.xebialabs.deployit.plugin.api.udm.Environment> allEnvs = newArrayList();

        allEnvs.addAll(ctx.getRepository().<com.xebialabs.deployit.plugin.api.udm.Environment>search(Type.valueOf(com.xebialabs.deployit.plugin.api.udm.Environment.class), ENVIRONMENTS_ROOT));
        allEnvs.addAll(ctx.getRepository().<Environment>search(Type.valueOf(Environment.class), ENVIRONMENTS_ROOT));

        for (ConfigurationItem instance : targetEnv.getLinkedTopLevelContainers()) {
            for (com.xebialabs.deployit.plugin.api.udm.Environment anEnv : allEnvs) {
                if (anEnv.equals(targetEnv)) {
                    continue;
                }

                for (Container container : anEnv.getMembers()) {
                    if (container.getId().startsWith(instance.getId())) {
                        ctx.logError("Environment " + anEnv.getId() + " contains " + container.getId() + " as a member.");
                        ctx.logError("Please remove this reference before " + targetEnv.getId() + " is destroyed.");
                        errors = true;
                    }
                }
            }
        }

        return errors ? StepExitCode.FAIL : StepExitCode.SUCCESS;
    }

    // Helpers

    /**
     * Tests whether container belongs to one of the cloud instances of targetEnv.
     */
    private boolean belongsToOneOfTopLevelContainers(final Container container, final Environment targetEnv) {
        System.out.println("container " + container.getId() + " targetEnv" + targetEnv.getId());
        return Collections2.filter(targetEnv.getLinkedTopLevelContainers(), new Predicate<Container>() {
            @Override
            public boolean apply(final Container input) {
                System.out.println("container " + container.getId() + " input " + input.getId());
                return container.equals(input) || container.getId().startsWith(input.getId() + "/");
            }
        }).size() == 1;
    }
}
