package com.xebialabs.deployit.plugin.cloud.step;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.google.common.base.Function;
import com.google.common.base.Predicate;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.cloud.ci.EnvironmentTemplate;
import com.xebialabs.deployit.plugin.cloud.step.rule.SingleEnvCheck;

import freemarker.template.TemplateException;

import static com.google.common.collect.Collections2.filter;
import static com.google.common.collect.Collections2.transform;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.plugin.cloud.util.ContextHelper.wrapped;
import static com.xebialabs.deployit.plugin.cloud.util.MapsHelper.putOrMerge;
import static java.util.Collections.emptyList;

public class ValidateEnvironmentDescriptorStep extends ValidateDescriptorStep {

    public static final Function<ConfigurationItem,String> CI_TO_ID = new Function<ConfigurationItem, String>() {
        @Override
        public String apply(ConfigurationItem input) {
            return input.getId();
        }
    };

    public ValidateEnvironmentDescriptorStep(EnvironmentTemplate environmentTemplate, String ciName) {
        super(environmentTemplate.getXmlDescriptor(), ciName, new SingleEnvCheck());

        scope.put("environmentId", "Environments/exampleEnvironment");
        scope.put("hosts", emptyList());
        scope.put("containers", emptyList());
        scope.put("environmentTemplate", environmentTemplate);
    }

    @Override
    protected String resolveTemplate(final String tpl, final Map<Object, Object> initialScope, ExecutionContext ctx) throws TemplateException {
        List<ConfigurationItem> parsedCis = wrapped(ctx).safeGet(ContextAttribute.PARSED_CIS, Collections.<ConfigurationItem>emptyList());

        Collection<ConfigurationItem> hostCis = filter(parsedCis, new Predicate<ConfigurationItem>() {
            @Override
            public boolean apply(ConfigurationItem input) {
                Type cloudSshHost = Type.valueOf("cloud.SshHost");
                Type cloudCifsHost = Type.valueOf("cloud.CifsHost");
                return input.getType().instanceOf(cloudSshHost) || input.getType().instanceOf(cloudCifsHost);
            }
        });

        Collection<ConfigurationItem> containerCis = filter(parsedCis, new Predicate<ConfigurationItem>() {
            @Override
            public boolean apply(ConfigurationItem input) {
                return input.getType().instanceOf(Type.valueOf("udm.Container"));
            }
        });

        return descriptorResolver.resolve(tpl, putOrMerge(putOrMerge(initialScope, "hosts", hostCis), "containers", containerCis));
    }

    @Override
    protected List<ConfigurationItem> parseCis(final String tpl, final ExecutionContext ctx) {
        List<ConfigurationItem> parsedCis = wrapped(ctx).safeGet(ContextAttribute.PARSED_CIS, Collections.<ConfigurationItem>emptyList());

        return ciParser.fromString(tpl, ctx.getRepository(), newHashSet(transform(parsedCis, CI_TO_ID)));
    }
}
