from ec2.ec2_helper import EC2Helper


class EC2ElasticIpHelper(EC2Helper):
    def __init__(self, deployed):
        super(EC2ElasticIpHelper, self).__init__(deployed)

    def create_elastic_ip(self):
        response = self.ec2_client.allocate_address(DryRun=False, Domain=self.deployed.elasticIpDomain)
        if response['ResponseMetadata']['HTTPStatusCode'] != 200:
            raise Exception("elastic-ip could not be created")
        elif self.deployed.elasticIpDomain == 'vpc':
            self.deployed.elasticIpAllocationId = response['AllocationId']
        return response['PublicIp']

    def associate_elastic_ip(self, elastic_ip, allocation_id=None):
        params = {'InstanceId': self.deployed.instanceId}
        if allocation_id is not None:
            params['AllocationId'] = allocation_id
            if self.is_multiple_interface():
                del params['InstanceId']
                params['NetworkInterfaceId'] = self.get_network_interface_at_zero_index()
        else:
            params['PublicIp'] = elastic_ip
        response = self.ec2_client.associate_address(**params)
        if response['ResponseMetadata']['HTTPStatusCode'] != 200:
            raise Exception("elastic-ip could not be associated")

    def release_elastic_ip(self, elastic_ip, allocation_id=None):
        params = {}
        if allocation_id is not None:
            params['AllocationId'] = allocation_id
        else:
            params['PublicIp'] = elastic_ip

        response = self.ec2_client.release_address(**params)
        if response['ResponseMetadata']['HTTPStatusCode'] != 200:
            raise Exception("elastic-ip could not be released")

    def disassociate_elastic_ip(self, elastic_ip, association_id=None):
        params = {}
        if association_id is not None:
            params['AssociationId'] = association_id
        else:
            params['PublicIp'] = elastic_ip
        response = self.ec2_client.disassociate_address(**params)
        if response['ResponseMetadata']['HTTPStatusCode'] != 200:
            raise Exception("elastic-ip could not be associated")

    def get_association_id(self, elastic_ip):
        response = self.ec2_client.describe_addresses(
            PublicIps=[elastic_ip])
        return response['Addresses'][0]['AssociationId']

    def is_multiple_interface(self):
        return len(self.instance.network_interfaces) > 1

    def get_network_interface_at_zero_index(self):
        for ni in self.instance.network_interfaces:
           if ni.attachment['DeviceIndex'] == 0:
               return ni.network_interface_id

    def get_public_ip(self):
        response = self.get_describe_instance_response()
        public_ip = response['Reservations'][0]['Instances'][0]['PublicIpAddress']
        return public_ip

    def get_public_hostname(self):
        response = self.get_describe_instance_response()
        return response['Reservations'][0]['Instances'][0]['PublicDnsName']

    def get_describe_instance_response(self):
        response = self.ec2_client.describe_instances(
            InstanceIds=[
                self.deployed.instanceId
            ])
        return response
