from auto_scaling.group.helpers.elb_helper import ElbHelper
from auto_scaling.group.helpers.policy_helper import PolicyHelper
from auto_scaling.group.helpers.tags_helper import TagsHelper
from auto_scaling.group.helpers.target_group_helper import ASGTargetGroupHelper
from auto_scaling.group.helpers.vpc_helper import VpcHelper


class AutoScalingValidator:
    def __init__(self, deployed, previous_deployed=None):
        self.__previous_deployed = previous_deployed
        self.__deployed = deployed
        self.__tag_helper = TagsHelper(self.__deployed, self.__previous_deployed)
        self.__elb_helper = ElbHelper(self.__deployed, self.__previous_deployed)
        self.__tg_helper = ASGTargetGroupHelper(self.__deployed, self.__previous_deployed)
        self.__vpc_helper = VpcHelper(self.__deployed, self.__previous_deployed)
        self.__policy_helper = PolicyHelper(self.__deployed, self.__previous_deployed)

    def are_resources_associated(self):
        return self.__elb_helper.are_new_load_balancers_attached() or \
               self.__tg_helper.are_new_target_groups_attached() or \
               self.__vpc_helper.are_new_availability_zones_attached() or \
               self.__vpc_helper.are_new_subnets_attached() or \
               self.is_changed("launchConfigurationInstanceId")

    def are_resources_dissociated(self):
        return self.__elb_helper.are_load_balancers_detached() or \
               self.__tg_helper.are_target_groups_detached() or \
               self.__vpc_helper.are_availability_zones_detached() or \
               self.__vpc_helper.are_subnets_detached()

    def are_properties_modified(self):
        return bool(self.__tag_helper.get_tags_to_be_deleted()) or \
               bool(self.__tag_helper.get_tags_to_be_updated()) or \
               self.__are_attributes_modified() or \
               self.__policy_helper.are_new_termination_policies_attached() or \
               self.__policy_helper.are_termination_policies_detached()

    def should_not_modify_subnets(self):
        if self.__vpc_helper.are_subnets_modified():
            raise RuntimeError(
                "You are not allowed to change the subnets associated with the auto scaling group")

    def should_not_modify_availability_zones(self):
        if self.__vpc_helper.are_availability_zones_modified():
            raise RuntimeError(
                "You are not allowed to change the availability zones associated with the auto scaling group")

    def should_not_specify_both_instance_id_and_image(self):
        if self.__deployed.launchConfigurationInstanceId \
                and (self.__deployed.launchConfigurationAMIId or self.__deployed.launchConfigurationInstanceType):
            raise RuntimeError("Both InstanceID and AMI ID/Instance Type can not be specified.")

    def __are_attributes_modified(self):
        properties = [
            "minSize",
            "maxSize",
            "desiredCapacity",
            "coolDown",
            "newInstancesProtectedFromScaleIn",
            "propagateAtLaunch",
            "resourceType",
            "healthCheckType",
            "healthCheckGracePeriod"
        ]
        return bool(filter(lambda property: self.is_changed(property), properties))

    def is_changed(self, field_name):
        return self.__deployed.getProperty(field_name) != self.__previous_deployed.getProperty(field_name)

    def is_instance_image_details_changed(self):
        return self.is_changed('launchConfigurationInstanceId') or self.is_changed("launchConfigurationAMIId") \
            or self.is_changed("launchConfigurationInstanceType") or self.is_changed("launchConfigurationUserData") \
            or self.is_changed("launchConfigurationSecurityGroups")


