from cloudformation.utils.resource_mapper import ResourceMapper


class EKSMapper(ResourceMapper):
    def __init__(self, client):
        super(EKSMapper, self).__init__(["k8s.Master", "k8s.Namespace"])
        self.eks_client = client
        # k8s.Master property => EKS Cluster property
        self.propertyMapper = {'clusterName':  'name',
                               'apiServerURL': 'endpoint'}

    def create_cis(self, cf_resource, cloud):
        if self.types_supported(cf_resource['PhysicalResourceId']):
            print("Creating CI of type 'k8s.Master' from '%s'" % cf_resource['PhysicalResourceId'])
            eks_cluster = self.eks_client.describe_cluster(name=cf_resource['PhysicalResourceId'])['cluster']
            cloud_id = cloud.id
            id = cloud_id[:cloud_id.rfind('/')+1] + cf_resource['PhysicalResourceId'] + "-EKSCluster"
            properties = {'isEKS':        True,
                          'accessKey':    cloud.getProperty('accesskey'),
                          'accessSecret': cloud.getProperty('accessSecret'),
                          'skipTLS':      True}
            for property in self.propertyMapper:
                properties[property] = eks_cluster[self.propertyMapper[property]]

            return [
                super(EKSMapper, self)._create_ci("k8s.Master", id, properties),
                self.create_namespace(id, "default"),
                self.create_namespace(id, "kube-system")
            ]
        else:
            return None

    def create_namespace(self, cluster_id, name):
        namespace_id = "%s/%s" % (cluster_id, name)
        return super(EKSMapper, self)._create_ci("k8s.Namespace", namespace_id, {'namespaceName': name})
