from rds.rds_helper import RDSHelper


class RDSSecurityGroupHelper(RDSHelper):
    def __init__(self, deployed):
        super(RDSSecurityGroupHelper, self).__init__(deployed)

    def create_security_group(self):
        return self.rds_client.create_db_security_group(DBSecurityGroupName=self.get_sg_name(),
                                                        DBSecurityGroupDescription=self.deployed.description)

    def destroy_security_group(self):
        return self.rds_client.delete_db_security_group(DBSecurityGroupName=self.get_sg_name())

    def attach_cidr_authorizations(self, cidr_ips):
        for cidr in cidr_ips:
            print "Attaching CIDR {0} to DB security group {1}".format(cidr, self.get_sg_name())
            self.rds_client.authorize_db_security_group_ingress(DBSecurityGroupName=self.get_sg_name(),
                                                                CIDRIP=cidr)

    def attach_ec2sg_authorizations(self, ec2_security_groups):
        for ec2sg in ec2_security_groups:
            print "Attaching EC2 Security group {0} to DB security group {1}".format(ec2sg, self.get_sg_name())
            self.rds_client.authorize_db_security_group_ingress(**self.__get_ec2_sg_param(ec2sg))


    def detach_cidr_authorizations(self, cidr_ips):
        for cidr in cidr_ips:
            print "Dettaching CIDR {0} from DB security group {1}".format(cidr, self.get_sg_name())
            self.rds_client.revoke_db_security_group_ingress(DBSecurityGroupName=self.get_sg_name(), CIDRIP=cidr)


    def detach_ec2sg_authorizations(self, ec2_security_groups):
        for ec2sg in ec2_security_groups:
            print "Detaching EC2 Security group {0} from DB security group {1}".format(ec2sg, self.get_sg_name())
            self.rds_client.revoke_db_security_group_ingress(**self.__get_ec2_sg_param(ec2sg))


    def is_all_authorizations_attached(self):
        response = self.rds_client.describe_db_security_groups(DBSecurityGroupName=self.get_sg_name())['DBSecurityGroups'][0]
        return self.__is_all_cidrip_authorized(response) and self.__is_all_ec2_security_group_authorized(response)


    def is_all_authorizations_detached(self, cidr_ips, ec2_sgs):
        response = self.rds_client.describe_db_security_groups(DBSecurityGroupName=self.get_sg_name())['DBSecurityGroups'][0]
        return self.__is_all_cidrip_detached(security_group=response, cidr_ips=cidr_ips) and self.__is_all_ec2_security_group_detached(security_group=response, ec2_sgs=ec2_sgs)


    def validate_security_group_modification(self, previous_deployed):
        if self.deployed.securityGroupName != previous_deployed.securityGroupName:
            raise RuntimeError("Name of DB security group cannot be updated once created.")
        if self.deployed.description != previous_deployed.description:
            raise RuntimeError("Description of DB security group cannot be updated once created.")


    def __get_ec2_sg_param(self, ec2_sg):
        param = {'DBSecurityGroupName': self.get_sg_name(),
                 'EC2SecurityGroupOwnerId': self.__get_account_id()}
        if self.is_starts_with_name(ec2_sg):
            param.update({'EC2SecurityGroupName': self.get_property_name(ec2_sg)})
        else:
            param.update({'EC2SecurityGroupId': ec2_sg})
        return param


    def __get_account_id(self):
        client = self.get_aws_client(resource_name="sts", region=self.deployed.region)
        return client.get_caller_identity()["Account"]

    def __get_property_name(self, prop):
        if self.is_starts_with_name(prop):
            return self.get_property_name(prop).lower()
        return prop


    def __is_all_ec2_security_group_detached(self,security_group, ec2_sgs):
        ec2_sgs = map(self.__get_property_name, ec2_sgs)
        if len(filter(lambda sg: sg['EC2SecurityGroupName'].lower() in ec2_sgs or sg['EC2SecurityGroupId'] in ec2_sgs , security_group['EC2SecurityGroups'])) > 0:
            return False
        return True

    def get_sg_name(self):
        return self.deployed.securityGroupName if self.deployed.securityGroupName else self.deployed.name

    @staticmethod
    def __is_all_cidrip_detached(security_group, cidr_ips):
        if len(filter(lambda cidr_ip: cidr_ip['CIDRIP'] in cidr_ips, security_group['IPRanges'])) > 0:
            return False
        return True


    @staticmethod
    def __is_all_cidrip_authorized(cidr_ips):
        if len(filter(lambda sg: sg['Status'] != 'authorized', cidr_ips['IPRanges'])) > 0:
            return False
        return True


    @staticmethod
    def __is_all_ec2_security_group_authorized(security_group):
        if len(filter(lambda sg: sg['Status'] != 'authorized', security_group['EC2SecurityGroups'])) > 0:
            return False
        return True
