from ec2.ec2_helper import EC2Helper
from ec2.ni.ni_helper import NIHelper

class EC2InstanceHelper(EC2Helper):
    ni_device_index_0 = '0'

    def __init__(self, deployed):
        super(EC2InstanceHelper, self).__init__(deployed)

    def get_ec2_params(self):
        ami_id_name = self.deployed.amiId
        if self.is_starts_with_name(ami_id_name):
            ami_id_name = self.get_image_id_by_name(self.get_property_name(ami_id_name))

        params_dict = {'ImageId': ami_id_name, 'MinCount': 1, 'MaxCount': 1,
                       'InstanceType': self.deployed.instanceType}
        if self.deployed.metadataAccessibleInfo == 'Enabled':
            metadata_options = {
                'HttpEndpoint': 'enabled',
                'HttpTokens': 'required',
                'HttpPutResponseHopLimit': self.deployed.metadataResponseHopLimit
            }
            params_dict['MetadataOptions'] =  metadata_options

        placement = {}
        if self.deployed.subnet:
            if self.is_starts_with_name(self.deployed.subnet):
                params_dict['SubnetId'] = self.get_subnet_id_by_name(self.get_property_name(self.deployed.subnet))
            else:
                params_dict['SubnetId'] = self.deployed.subnet
        if self.deployed.securityGroup and not self.is_ni_with_0_index():
            sg_ids = self.get_security_group_id_list(self.deployed.securityGroup)
            if len(sg_ids) > 0:
                params_dict['SecurityGroupIds'] = sg_ids
        if self.deployed.shutdownBehavior:
            params_dict['InstanceInitiatedShutdownBehavior'] = self.deployed.shutdownBehavior
        if self.deployed.terminationProtection:
            params_dict['DisableApiTermination'] = self.deployed.terminationProtection
        if self.deployed.userData:
            params_dict['UserData'] = self.deployed.userData
        if self.deployed.detailedMonitoring:
            params_dict['Monitoring'] = {'Enabled': self.deployed.detailedMonitoring}
        if self.deployed.tenancy:
            placement.update({'Tenancy': self.deployed.tenancy})
        if self.deployed.iAmRoleARN:
            params_dict['IamInstanceProfile'] = {'Arn': self.deployed.iAmRoleARN}
        if self.deployed.availabilityZone:
            placement.update({'AvailabilityZone': self.deployed.availabilityZone})
        if len(placement) > 0:
            params_dict['Placement'] = placement
        if self.is_ni_with_0_index():
            network_interface = self.deployed.networkInterfaces[self.ni_device_index_0]
            if network_interface.lower().startswith('name:'):
                ni_helper = NIHelper(self.deployed)
                network_interface = ni_helper.get_network_interface_id(network_interface[5:])
            params_dict['NetworkInterfaces'] = [{'NetworkInterfaceId': network_interface, 'DeviceIndex': 0}]
        if self.deployed.keyName:
            params_dict['KeyName'] = self.deployed.keyName
        if self.deployed.createVolumeOnInstanceCreation:
            blockDeviceMappings = []
            Ebsdict = {}
            blockDeviceMappingsDict = {}
            if self.deployed.deleteOnTermination:
                Ebsdict['DeleteOnTermination'] = self.deployed.deleteOnTermination
            if self.deployed.iops:
                Ebsdict['Iops'] = self.deployed.iops
            if self.deployed.size:
                Ebsdict['VolumeSize'] = self.deployed.size
            if self.deployed.snapshotId:
                Ebsdict['SnapshotId'] = self.deployed.snapshotId
            if self.deployed.volumeType:
                Ebsdict['VolumeType'] = self.deployed.volumeType
            if self.deployed.kmsKeyId:
                Ebsdict['KmsKeyId'] = self.deployed.kmsKeyId
            if self.deployed.encryption:
                Ebsdict['Encrypted'] = self.deployed.encryption
            if self.deployed.throughput:
                Ebsdict['Throughput'] = self.deployed.throughput
            if self.deployed.outpostArn:
                Ebsdict['OutpostArn'] = self.deployed.outpostArn
            if self.deployed.deviceName:
                blockDeviceMappingsDict['DeviceName'] = self.deployed.deviceName
            if self.deployed.virtualName:
                blockDeviceMappingsDict['VirtualName'] = self.deployed.virtualName
            if self.deployed.noDevice:
                blockDeviceMappingsDict['NoDevice'] = self.deployed.noDevice
            if Ebsdict:
                blockDeviceMappingsDict['Ebs'] = Ebsdict
            blockDeviceMappings = blockDeviceMappingsDict
            params_dict['BlockDeviceMappings'] = [blockDeviceMappings]
        return params_dict

    def create_instance(self):
        if self.deployed.amiId is None:
            self.deployed.amiId = self.deployed.deployable.name
        ec2_instance_params = self.get_ec2_params()
        instances = self.ec2.create_instances(**ec2_instance_params)
        if len(instances) > 0:
            self.instance = instances[0]

            print "instance {0} is in {1} state".format(self.instance.instance_id,
                                                        self.instance.state.get("Name"))
            return self.instance
        else:
            raise Exception("Not able to create AWS instance")

    def attach_security_groups(self):
        self.instance.modify_attribute(Groups=self.get_security_group_id_list(self.deployed.securityGroup))

    def is_instance_running(self, instance_id):
        instance_state_dict = self.ec2_client.describe_instance_status(
            InstanceIds=[
                instance_id,
            ],
            Filters=[{'Name': 'instance-state-name', 'Values': ['running']},
                     {'Name': 'instance-status.reachability', 'Values': ['passed']},
                     {'Name': 'instance-status.status', 'Values': ['ok']},
                     {'Name': 'system-status.status', 'Values': ['ok']}
                     ])
        statuses = instance_state_dict.get("InstanceStatuses")
        if statuses:
            print("instance {0} is in {1} state".format(instance_id, statuses[0]['InstanceState']['Name']))
            self.instance.reload
            return True
        else:
            return False

    def is_instance_stopped(self, instance_id):
        instance_state_dict = self.ec2_client.describe_instance_status(
            InstanceIds=[
                instance_id,
            ],
            Filters=[{'Name': 'instance-state-name', 'Values': ['stopped']}],
            IncludeAllInstances=True)
        statuses = instance_state_dict.get("InstanceStatuses")
        if statuses:
            print("instance {0} is in {1} state".format(instance_id, statuses[0]['InstanceState']['Name']))
            return True
        else:
            return False

    def get_current_status(self, instance_id):
        instance_state_dict = self.ec2_client.describe_instance_status(
            InstanceIds=[
                instance_id,
            ],
            IncludeAllInstances=True)
        statuses = instance_state_dict.get("InstanceStatuses")
        if statuses:
            print("instance {0} is in {1} state".format(instance_id, statuses[0]['InstanceState']['Name']))
            return statuses[0]['InstanceState']['Name']
        else:
            print("Unable to Get Instance Status")
            return None

    def destroy_instance(self):
        ids = [self.deployed.instanceId]
        return self.ec2.instances.filter(InstanceIds=ids).terminate()

    def associate_iam_instance_profile(self, instance_id):
        self.ec2_client.associate_iam_instance_profile(
            IamInstanceProfile={'Arn': self.deployed.iAmRoleARN},
            InstanceId=instance_id
        )

    def modify_iam_instance_profile(self, instance_id):
        association_id = self.get_iam_instance_profile_association_id(instance_id)
        self.ec2_client.replace_iam_instance_profile_association(
            IamInstanceProfile={'Arn': self.deployed.iAmRoleARN},
            AssociationId=association_id)

    def get_iam_instance_profile_association_id(self, instance_id):
        response = self.ec2_client.describe_iam_instance_profile_associations(
            Filters=[{'Name': 'instance-id', 'Values': [instance_id]},
                     {'Name': 'state', 'Values': ['associating', 'associated']}]
        )
        return response['IamInstanceProfileAssociations'][0]['AssociationId']

    def modify_instance_attribute(self, instance_id, attribute, value):
        attr_dict = {'InstanceId': instance_id, attribute: {'Value': value}}
        modify_response = self.ec2_client.modify_instance_attribute(**attr_dict)
        print "Property %s updated for EC2 instance %s" % (attribute, instance_id)
        return modify_response

    def enable_detailed_monitoring(self, instance_id):
        self.ec2_client.monitor_instances(InstanceIds=[instance_id])
        print "Detailed monitoring enabled EC2 instance %s" % (instance_id)

    def disable_detailed_monitoring(self, instance_id):
        self.ec2_client.unmonitor_instances(InstanceIds=[instance_id])
        print "Detailed monitoring disabled EC2 instance %s" % (instance_id)

    def modify_tenancy(self, instance_id, tenancy):
        self.ec2_client.modify_instance_placement(
            InstanceId=instance_id,
            Tenancy=tenancy)

    def stop_ec2_instance(self, instance_id):
        stop_response = self.ec2_client.stop_instances(
            InstanceIds=[instance_id])
        return stop_response

    def start_ec2_instance(self, instance_id):
        start_response = self.ec2_client.start_instances(
            InstanceIds=[instance_id])
        return start_response

    def get_image_id_by_name(self, image_name):
        image_response = self.ec2_client.describe_images(Filters=[
            {
                'Name': 'tag:Name',
                'Values': [image_name]
            },
        ])
        if image_response['ResponseMetadata']['HTTPStatusCode'] != 200 or len(image_response['Images']) == 0:
            raise Exception("AMI with name %s not found in AWS." % image_name)
        elif len(image_response['Images']) > 1:
            raise Exception("More than one AMI found with name %s." % image_name)
        else:
            return image_response['Images'][0]['ImageId']

    def attach_volume(self, volume_id, device):
        return self.ec2_client.attach_volume(DryRun=False,
                                             VolumeId=volume_id,
                                             InstanceId=self.deployed.instanceId,
                                             Device=device)

    def is_instance_destroyed(self):
        if self.destroy_instance()[0].get('TerminatingInstances')[0].get('CurrentState').get('Name') == 'terminated':
            return True
        return False

    def attach_network_interface(self, device_index, network_interface_id):
        self.ec2_client.attach_network_interface(
            NetworkInterfaceId=network_interface_id,
            InstanceId=self.deployed.instanceId,
            DeviceIndex=device_index
        )

    def get_attachment_id(self, network_interface_id):
        response = self.ec2_client.describe_network_interfaces(
            NetworkInterfaceIds=[
                network_interface_id
            ])
        if response['ResponseMetadata']['HTTPStatusCode'] != 200 or len(response['NetworkInterfaces']) == 0:
            raise RuntimeError("No network interfaces found with id %s" % network_interface_id)

        return response['NetworkInterfaces'][0]['Attachment']['AttachmentId']

    def detach_network_interface(self, network_interface_id):
        attachment_id = self.get_attachment_id(network_interface_id)
        self.ec2_client.detach_network_interface(
            AttachmentId=attachment_id,
            Force=True
        )

    def is_network_interface_detached(self, ni_id):
        response = self.ec2_client.describe_network_interfaces(DryRun=False,
                                                               Filters=[{'Name': 'network-interface-id', 'Values': [ni_id]}])
        if response['ResponseMetadata']['HTTPStatusCode'] != 200 or len(response['NetworkInterfaces']) == 0:
            raise RuntimeError("No network interfaces found with id %s" % ni_id)
        elif len(response['NetworkInterfaces']) > 1:
            raise RuntimeError("Multiple network interfaces found with id %s" % ni_id)

        if response['NetworkInterfaces'][0]['Status'] == 'available':
            return True
        else:
            return False

    def is_ni_with_0_index(self):
        return self.ni_device_index_0 in self.deployed.networkInterfaces

