import commons
from botocore.session import Session as BotocoreSession

from boto3.session import Session


class EC2Helper(object):
    def __init__(self, deployed):
        self.deployed = deployed

        botocore_session = BotocoreSession()
        botocore_session.lazy_register_component('data_loader',
                                                 lambda: commons.create_loader())

        self.session = Session(aws_access_key_id=deployed.container.accesskey,
                               aws_secret_access_key=deployed.container.accessSecret,
                               botocore_session=botocore_session)
        self.ec2 = self.session.resource('ec2', region_name=deployed.region, use_ssl=False)
        self.ec2_client = self.session.client('ec2', region_name=deployed.region, use_ssl=False)
        if hasattr(deployed, 'instanceId') and deployed.instanceId is not None:
            self.instance = self.ec2.Instance(deployed.instanceId)

    def get_security_group_id_list(self, security_groups):
        sg_ids = []
        for sg in security_groups:
            if self.is_starts_with_name(sg):
                sg_ids.append(self.get_security_group_id(self.get_property_name(sg)))
            else:
                sg_ids.append(sg)
        return sg_ids

    def get_security_group_id(self, security_group):
        sg_response = self.ec2_client.describe_security_groups(Filters=[
            {
                'Name': 'tag:Name',
                'Values': [security_group]
            },
        ])
        if sg_response['ResponseMetadata']['HTTPStatusCode'] != 200 or len(sg_response['SecurityGroups']) == 0:
            raise Exception("Security group %s not found in AWS." % security_group)
        elif len(sg_response['SecurityGroups']) > 1:
            raise Exception("More than one security groups found with name %s." % security_group)
        else:
            return sg_response['SecurityGroups'][0]['GroupId']

    def is_starts_with_name(self, property_value):
        return property_value.lower().startswith('name:')

    def get_property_name(self,property_name):
        return property_name[5:]

    def create_elastic_ip(self):
        response = self.ec2_client.allocate_address(DryRun=False, Domain='standard')
        if response['ResponseMetadata']['HTTPStatusCode'] != 200:
            raise Exception("elastic-ip could not be created")
        else:
            return response['PublicIp']

    def associate_elastic_ip(self, elastic_ip):
        response = self.ec2_client.associate_address(DryRun=False, InstanceId=self.deployed.instanceId, PublicIp=elastic_ip)
        if response['ResponseMetadata']['HTTPStatusCode'] != 200:
            raise Exception("elastic-ip could not be associated")

    def release_elastic_ip(self):
        response = self.ec2_client.release_address(DryRun=False, PublicIp=self.deployed.publicIp)
        if response['ResponseMetadata']['HTTPStatusCode'] != 200:
            raise Exception("elastic-ip could not be released")

    def public_ip(self):
            return self.instance.public_ip_address

    def public_hostname(self):
        return self.instance.public_dns_name

    def get_subnet_id_by_name(self, subnet_name):
        response = self.ec2_client.describe_subnets(Filters=[
            {
                'Name': 'tag:Name',
                'Values': [subnet_name]
            },
        ])
        if response['ResponseMetadata']['HTTPStatusCode'] != 200 or len(response['Subnets']) == 0:
            raise Exception("Subnet with name %s not found in AWS." % subnet_name)
        elif len(response['Subnets']) > 1:
            raise Exception("More than one subnet found with name %s." % subnet_name)
        else:
            return response['Subnets'][0]['SubnetId']
