import commons
from botocore.session import Session as BotocoreSession

from boto3.session import Session


class EC2Helper(object):
    def __init__(self, deployed):
        self.deployed = deployed

        botocore_session = BotocoreSession()
        botocore_session.lazy_register_component('data_loader',
                                                 lambda: commons.create_loader())

        self.session = Session(aws_access_key_id=deployed.container.accesskey,
                               aws_secret_access_key=deployed.container.accessSecret,
                               botocore_session=botocore_session)
        self.ec2 = self.session.resource('ec2', region_name=deployed.region, use_ssl=False)
        self.ec2_client = self.session.client('ec2', region_name=deployed.region, use_ssl=False)
        if hasattr(deployed, 'instanceId') and deployed.instanceId is not None:
            self.instance = self.ec2.Instance(deployed.instanceId)

    def get_security_group_id_list(self, security_groups):
        sg_ids = []
        for sg in security_groups:
            if self.is_starts_with_name(sg):
                sg_ids.append(self.get_security_group_id(self.get_property_name(sg)))
            else:
                sg_ids.append(sg)
        return sg_ids

    def get_security_group_id(self, security_group):
        sg_response = self.ec2_client.describe_security_groups(Filters=[
            {
                'Name': 'tag:Name',
                'Values': [security_group]
            },
        ])
        if sg_response['ResponseMetadata']['HTTPStatusCode'] != 200 or len(sg_response['SecurityGroups']) == 0:
            raise Exception("Security group %s not found in AWS." % security_group)
        elif len(sg_response['SecurityGroups']) > 1:
            raise Exception("More than one security groups found with name %s." % security_group)
        else:
            return sg_response['SecurityGroups'][0]['GroupId']

    def is_starts_with_name(self, property_value):
        return property_value.lower().startswith('name:') if property_value else False

    def get_property_name(self,property_name):
        return property_name[5:]

    def public_ip(self):
        return self.instance.public_ip_address

    def public_hostname(self):
        return self.instance.public_dns_name

    def get_subnet_id_by_name(self, subnet_name):
        response = self.ec2_client.describe_subnets(Filters=[
            {
                'Name': 'tag:Name',
                'Values': [subnet_name]
            },
        ])
        if response['ResponseMetadata']['HTTPStatusCode'] != 200 or len(response['Subnets']) == 0:
            raise Exception("Subnet with name %s not found in AWS." % subnet_name)
        elif len(response['Subnets']) > 1:
            raise Exception("More than one subnet found with name %s." % subnet_name)
        else:
            return response['Subnets'][0]['SubnetId']

    def get_subnet_ids_by_names(self, subnet_names):
        if not subnet_names or len(subnet_names) == 0:
            return []
        response = self.ec2_client.describe_subnets(Filters=[
            {
                'Name': 'tag:Name',
                'Values': subnet_names
            },
        ])
        if response['ResponseMetadata']['HTTPStatusCode'] != 200 or not response['Subnets']:
            raise Exception("Unable to to fetch subnets from AWS.")
        else:
            return [subnet['SubnetId'] for subnet in response['Subnets']]

    def get_network_interface_id(self, ni_name):
        response = self.ec2_client.describe_network_interfaces(DryRun=False,
                                                               Filters=[{'Name': 'tag:Name', 'Values': [ni_name]}])
        if response['ResponseMetadata']['HTTPStatusCode'] != 200 or len(response['NetworkInterfaces']) == 0:
            raise RuntimeError("No network interfaces found with name %s" % ni_name)
        elif len(response['NetworkInterfaces']) > 1:
            raise RuntimeError("Multiple network interfaces found with name %s" % ni_name)

        return response['NetworkInterfaces'][0]['NetworkInterfaceId']

    def get_subnet_id_list(self, subnets):
        subnet_ids = []
        # Filter subnets which are already IDs and use them directly
        subnets_with_id = filter(lambda subnet: not self.is_starts_with_name(subnet), subnets)
        subnet_ids.extend(subnets_with_id)

        # Filter all the subnets which start with 'Name:' and resolve ids
        subnets_with_name = filter(lambda subnet: self.is_starts_with_name(subnet), subnets)
        subnet_names = [self.get_property_name(subnet) for subnet in subnets_with_name]
        subnet_ids.extend(self.get_subnet_ids_by_names(subnet_names))
        return subnet_ids

    def set_resource_name(self, resource_id, name):
        print "Setting name tag as {0} for resource {1}".format(name, resource_id)
        self.create_ec2_tag(resource_id, 'Name', name)

    def create_ec2_tag(self, resource_id, key, value):
        self.ec2_client.create_tags(DryRun=False, Resources=[resource_id],
                                    Tags=[{'Key': key, 'Value': value}])

    def delete_ec2_tag(self, resource_id, key, value):
        self.ec2_client.delete_tags(DryRun=False, Resources=[resource_id],
                                    Tags=[{'Key': key, 'Value': value}])
        print "Deleted tag %s, %s from resource" % (key, value)

    def get_network_interface_id_by_id_or_name(self, ni_identifier):
        if self.is_starts_with_name(ni_identifier):
            return self.get_network_interface_id(self.get_property_name(ni_identifier))
        else:
            return ni_identifier

    def get_instance_id_by_id_or_name(self, instance_identifier):
        if self.is_starts_with_name(instance_identifier):
            return self.get_instance_id(self.get_property_name(instance_identifier))
        else:
            return instance_identifier

    def get_instance_id(self, instance_name):
        response = self.ec2_client.describe_instances(Filters=[
            {
                'Name': 'tag:Name',
                'Values': [instance_name]
            },
        ])
        if response['ResponseMetadata']['HTTPStatusCode'] != 200 or len(response['Reservations']) == 0:
            raise Exception("Instance with name %s not found in AWS." % instance_name)
        elif len(response['Reservations']) > 1:
            raise Exception("More than one instance found with name %s." % instance_name)
        else:
            return response['Reservations'][0]['Instances'][0]['InstanceId']

    def get_vpc_peering_connection_id_by_id_or_name(self, vpc_peering_connection_identifier):
        if self.is_starts_with_name(vpc_peering_connection_identifier):
            return self.get_vpc_peering_connection_id(self.get_property_name(vpc_peering_connection_identifier))
        else:
            return vpc_peering_connection_identifier

    def get_vpc_peering_connection_id(self, vpc_peering_connection_name):
        response = self.ec2_client.describe_vpc_peering_connections(Filters=[
            {
                'Name': 'tag:Name',
                'Values': [vpc_peering_connection_name]
            },
        ])
        if response['ResponseMetadata']['HTTPStatusCode'] != 200 or len(response['VpcPeeringConnections']) == 0:
            raise Exception("VPC Peering Connection with name %s not found in AWS." % vpc_peering_connection_name)
        elif len(response['VpcPeeringConnections']) > 1:
            raise Exception("More than one VPC Peering Connections found with name %s." % vpc_peering_connection_name)
        else:
            return response['VpcPeeringConnections'][0]['VpcPeeringConnectionId']

    def get_internet_gateway_id_by_id_or_name(self, internet_gateway_identifier):
        if self.is_starts_with_name(internet_gateway_identifier):
            return self.get_internet_gateway_id(self.get_property_name(internet_gateway_identifier))
        else:
            return internet_gateway_identifier

    def get_internet_gateway_id(self, internet_gateway_name):
        response = self.ec2_client.describe_internet_gateways(Filters=[
            {
                'Name': 'tag:Name',
                'Values': [internet_gateway_name]
            },
        ])
        if response['ResponseMetadata']['HTTPStatusCode'] != 200 or len(response['InternetGateways']) == 0:
            raise Exception("Internet Gateway with name %s not found in AWS." % internet_gateway_name)
        elif len(response['InternetGateways']) > 1:
            raise Exception("More than one Internet Gateways found with name %s." % internet_gateway_name)
        else:
            return response['InternetGateways'][0]['InternetGatewayId']

