from commons.aws_helper import AWSHelper

class EC2Helper(AWSHelper):
    def __init__(self, deployed):
        super(EC2Helper, self).__init__(deployed)
        self.ec2 = self.session.resource('ec2', region_name=deployed.region)
        self.ec2_client = self.get_aws_client(region=deployed.region, resource_name='ec2')
        if hasattr(deployed, 'instanceId') and deployed.instanceId is not None:
            self.instance = self.ec2.Instance(deployed.instanceId)

    def get_security_group_id_list(self, security_groups):
        sg_ids = []
        for sg in security_groups:
            if self.is_starts_with_name(sg):
                sg_ids.append(self.get_security_group_id(self.get_property_name(sg)))
            else:
                sg_ids.append(sg)
        return sg_ids

    def get_security_group_id(self, security_group):
        sg_response = self.ec2_client.describe_security_groups(Filters=[
            {
                'Name': 'tag:Name',
                'Values': [security_group]
            },
        ])
        if not AWSHelper.is_success(sg_response) or not sg_response['SecurityGroups']:
            raise RuntimeError("Security group %s not found in AWS." % security_group)
        elif len(sg_response['SecurityGroups']) > 1:
            raise RuntimeError("More than one security groups found with name %s." % security_group)
        else:
            return sg_response['SecurityGroups'][0]['GroupId']

    def public_ip(self):
        return self.instance.public_ip_address

    def public_hostname(self):
        return self.instance.public_dns_name

    def get_subnet_id_by_name(self, subnet_name):
        response = self.ec2_client.describe_subnets(Filters=[
            {
                'Name': 'tag:Name',
                'Values': [subnet_name]
            },
        ])
        if not AWSHelper.is_success(response) or not response['Subnets']:
            raise RuntimeError("Subnet with name %s not found in AWS." % subnet_name)
        elif len(response['Subnets']) > 1:
            raise RuntimeError("More than one subnet found with name %s." % subnet_name)
        else:
            return response['Subnets'][0]['SubnetId']

    def get_subnet_ids_by_names(self, subnet_names):
        if not subnet_names:
            return []
        response = self.ec2_client.describe_subnets(Filters=[
            {
                'Name': 'tag:Name',
                'Values': subnet_names
            },
        ])
        if not AWSHelper.is_success(response) or not response['Subnets']:
            raise RuntimeError("Unable to to fetch subnets from AWS.")
        else:
            return [subnet['SubnetId'] for subnet in response['Subnets']]

    def get_network_interface_id(self, ni_name):
        response = self.ec2_client.describe_network_interfaces(DryRun=False,
                                                               Filters=[{'Name': 'tag:Name', 'Values': [ni_name]}])
        if not AWSHelper.is_success(response) or not response['NetworkInterfaces']:
            raise RuntimeError("No network interfaces found with name %s" % ni_name)
        elif len(response['NetworkInterfaces']) > 1:
            raise RuntimeError("Multiple network interfaces found with name %s" % ni_name)

        return response['NetworkInterfaces'][0]['NetworkInterfaceId']

    def get_subnet_id_list(self, subnets):
        subnet_ids = []
        # Filter subnets which are already IDs and use them directly
        subnets_with_id = filter(lambda subnet: not self.is_starts_with_name(subnet), subnets)
        subnet_ids.extend(subnets_with_id)

        # Filter all the subnets which start with 'Name:' and resolve ids
        subnets_with_name = filter(lambda subnet: self.is_starts_with_name(subnet), subnets)
        subnet_names = [self.get_property_name(subnet) for subnet in subnets_with_name]
        subnet_ids.extend(self.get_subnet_ids_by_names(subnet_names))
        return subnet_ids

    def set_resource_name(self, resource_id, name):
        print "Setting name for resource {0} to {1}".format(resource_id, name)
        self.create_ec2_tag(resource_id, 'Name', name)

    def create_ec2_tag(self, resource_id, key, value):
        self.ec2_client.create_tags(DryRun=False, Resources=[resource_id],
                                    Tags=[{'Key': key, 'Value': value}])
        print "Added tag %s, %s to resource" % (key, value)

    def delete_ec2_tag(self, resource_id, key, value):
        self.ec2_client.delete_tags(DryRun=False, Resources=[resource_id],
                                    Tags=[{'Key': key, 'Value': value}])
        print "Deleted tag %s, %s from resource" % (key, value)

    def get_network_interface_id_by_id_or_name(self, ni_identifier):
        if self.is_starts_with_name(ni_identifier):
            return self.get_network_interface_id(self.get_property_name(ni_identifier))
        else:
            return ni_identifier

    def get_instance_id_by_id_or_name(self, instance_identifier):
        if self.is_starts_with_name(instance_identifier):
            return self.get_instance_id(self.get_property_name(instance_identifier))
        else:
            return instance_identifier

    def get_instance_id(self, instance_name):
        running_instances = self.get_instance_ids([instance_name])
        if len(running_instances) > 1: raise RuntimeError(
            "More than one running instance found with name {}.".format(instance_name))
        return running_instances[0]

    def get_instance_ids(self, instance_names):
        instances = self.get_instances(instance_names)
        return map(lambda instance: instance['InstanceId'], instances)

    def get_instances(self, instance_names):
        response = self.ec2_client.describe_instances(Filters=[
            {
                'Name': 'tag:Name',
                'Values': instance_names
            },
        ])
        if not AWSHelper.is_success(response) or not response['Reservations']:
            raise RuntimeError("Instances with names {} not found in AWS.".format(instance_names))
        return self.__get_non_terminated_instances(response)

    def __get_non_terminated_instances(self, response):
        reservations = map(lambda reservation: reservation, response['Reservations'])
        all_instances = map(lambda reservation: reservation["Instances"], reservations)
        return filter(lambda instance: instance["State"]["Name"] != 'terminated', sum(all_instances, []))

    def get_vpc_peering_connection_id_by_id_or_name(self, vpc_peering_connection_identifier):
        if self.is_starts_with_name(vpc_peering_connection_identifier):
            return self.get_vpc_peering_connection_id(self.get_property_name(vpc_peering_connection_identifier))
        else:
            return vpc_peering_connection_identifier

    def get_vpc_peering_connection_id(self, vpc_peering_connection_name):
        response = self.ec2_client.describe_vpc_peering_connections(Filters=[
            {
                'Name': 'tag:Name',
                'Values': [vpc_peering_connection_name]
            },
        ])
        if not AWSHelper.is_success(response) or not response['VpcPeeringConnections']:
            raise RuntimeError("VPC Peering Connection with name %s not found in AWS." % vpc_peering_connection_name)
        elif len(response['VpcPeeringConnections']) > 1:
            raise RuntimeError(
                "More than one VPC Peering Connections found with name %s." % vpc_peering_connection_name)
        else:
            return response['VpcPeeringConnections'][0]['VpcPeeringConnectionId']

    def get_internet_gateway_id_by_id_or_name(self, internet_gateway_identifier):
        if self.is_starts_with_name(internet_gateway_identifier):
            return self.get_internet_gateway_id(self.get_property_name(internet_gateway_identifier))
        else:
            return internet_gateway_identifier

    def get_internet_gateway_id(self, internet_gateway_name):
        response = self.ec2_client.describe_internet_gateways(Filters=[
            {
                'Name': 'tag:Name',
                'Values': [internet_gateway_name]
            },
        ])
        if not AWSHelper.is_success(response) or not response['InternetGateways']:
            raise RuntimeError("Internet Gateway with name %s not found in AWS." % internet_gateway_name)
        elif len(response['InternetGateways']) > 1:
            raise RuntimeError("More than one Internet Gateways found with name %s." % internet_gateway_name)
        else:
            return response['InternetGateways'][0]['InternetGatewayId']

