from ec2.ec2_helper import EC2Helper
from ec2.vpc.vpc_helper import VPCHelper


class SNHelper(EC2Helper):
    def __init__(self, deployed):
        super(SNHelper, self).__init__(deployed)
        self.vpc_helper = VPCHelper(deployed)

    def create_subnet(self):
        params = self.get_subnet_params()
        response = self.ec2_client.create_subnet(**params)
        if self.is_success(response):
            self.deployed.subnetId = response['Subnet']['SubnetId']
        return response

    def delete_subnet(self):
        response = self.ec2_client.delete_subnet(
            SubnetId=self.deployed.subnetId
        )
        return response

    def modify_auto_assign_public_ip(self):
        response = self.ec2_client.modify_subnet_attribute(
            SubnetId=self.deployed.subnetId,
            MapPublicIpOnLaunch={
                'Value': self.deployed.mapPublicIpOnLaunch
            })
        return response

    def modify_auto_assign_ipv6_address(self):
        response = self.ec2_client.modify_subnet_attribute(
            SubnetId=self.deployed.subnetId,
            AssignIpv6AddressOnCreation={
                'Value': self.deployed.assignIpv6AddressOnCreation
            })
        return response

    def get_subnet_params(self):
        if self.is_starts_with_name(self.deployed.vpc):
            vpc_id = self.vpc_helper.get_vpc_id_by_name(self.get_property_name(self.deployed.vpc))
        else:
            vpc_id = self.deployed.vpc

        params_dict = {'VpcId': vpc_id}
        if self.deployed.cidrBlock:
            params_dict['CidrBlock'] = self.deployed.cidrBlock
        if self.deployed.ipv6CidrBlock:
            params_dict['Ipv6CidrBlock'] = self.__create_ipv6_subnet_cidr(vpc_id)
        if self.deployed.availabilityZone:
            params_dict['AvailabilityZone'] = self.deployed.availabilityZone
        return params_dict

    def is_subnet_available(self, subnet_id):
        try:
            subnets = self.ec2_client.describe_subnets(SubnetIds=[subnet_id])["Subnets"]
            if subnets[0]["State"] == "available":
                return True
        except:
            return False

    def __create_ipv6_subnet_cidr(self, vpc_id):
        vpc = self.vpc_helper.get_vpc_by_id(vpc_id)
        ipv6_vpc_cidr_set = vpc['Ipv6CidrBlockAssociationSet']
        if ipv6_vpc_cidr_set:
            if len(self.deployed.ipv6CidrBlock) == 2:
                ipv6_vpc_cidr = ipv6_vpc_cidr_set[0]['Ipv6CidrBlock']
                return '{}{}::{}'.format(self.__extract_ipv6_cidr_block_prefix(ipv6_vpc_cidr), self.deployed.ipv6CidrBlock.lower(), self.deployed.ipv6CidrBlockSuffix)
            else:
                return self.deployed.ipv6CidrBlock.lower()
        else:
            raise RuntimeError('Subnet with IPv6 can not be created as VPC is not IPv6 enabled')

    @staticmethod
    def __extract_ipv6_cidr_block_prefix(ipv6_vpc_cidr):
        # Removing suffix ::/56
        ip_prefix = ipv6_vpc_cidr[:-5]
        word_count = ip_prefix.count(':')

        # ::/56 accommodates ffff:ffff:ffff:ff space.
        missing_word_count = 3 - word_count
        while missing_word_count > 1:
            ip_prefix = '{}:{}'.format(ip_prefix, '0')
            missing_word_count -= 1

        if missing_word_count == 1:
            ip_prefix = '{}:{}'.format(ip_prefix, '00')
        else:
            ip_prefix = ip_prefix[:-2]
        return ip_prefix

