import os
import tempfile
from java.nio.file import Files, Paths, StandardCopyOption

import com.xebialabs.deployit.plugin.aws.support.BotoLoader as BotoLoader
import commons
from boto3.session import Session
from botocore.session import Session as BotocoreSession


class AWSHelper(object):

    def __init__(self, deployed):
        self.deployed = deployed

        botocore_session = BotocoreSession()
        botocore_session.lazy_register_component('data_loader',
                                                 lambda: commons.create_loader())

        self.session = Session(aws_access_key_id=deployed.container.accesskey,
                                    aws_secret_access_key=deployed.container.accessSecret,
                                    botocore_session=botocore_session)

        AWSHelper.set_ca_bundle_path()

    def get_aws_client(self, region, resource_name='ec2'):
        if not self.deployed.container.verifySSL:
            return self.session.client(resource_name, region_name=region, verify=False)
        return self.session.client(resource_name, region_name=region)

    @staticmethod
    def extract_file_from_jar(config_file):
        file_url = BotoLoader.getResourceBySelfClassLoader(config_file)
        if file_url:
            tmp_file, tmp_abs_path = tempfile.mkstemp()
            tmp_file.close()
            Files.copy(file_url.openStream(), Paths.get(tmp_abs_path), StandardCopyOption.REPLACE_EXISTING)
            return tmp_abs_path

        else:
            return None

    @staticmethod
    def set_ca_bundle_path():
        ca_bundle_path = AWSHelper.extract_file_from_jar("botocore/vendored/requests/cacert.pem")
        os.environ['REQUESTS_CA_BUNDLE'] = ca_bundle_path

    @staticmethod
    def is_starts_with_name(property_value):
        return property_value.lower().startswith('name:') if property_value else False

    @staticmethod
    def get_property_name(property_name):
        return property_name[5:]

    @staticmethod
    def is_success(response):
        return 299 >= response['ResponseMetadata']['HTTPStatusCode'] >= 200

    @staticmethod
    def remove_none_keys(dict):
        return {k: v for k, v in dict.iteritems() if v is not None}

    @staticmethod
    def remove_empty_and_none_values(dict):
        return {k: v for k, v in dict.iteritems() if (bool(v) if isinstance(v, (list, set)) else v is not None)}

    @staticmethod
    def get_current_retry_count(context, counter_name_suffix):
        counter_name_suffix = "current_retry_{0}".format(counter_name_suffix)
        current_retry_count = context.getAttribute(counter_name_suffix)
        current_retry_count = 1 if not current_retry_count else current_retry_count
        return current_retry_count

    @staticmethod
    def increment_retry_counter(context, counter_name_suffix):
        current_retry_count = AWSHelper.get_current_retry_count(context, counter_name_suffix)
        current_retry_count = current_retry_count + 1
        AWSHelper.set_current_retry_count(context, counter_name_suffix, current_retry_count)

    @staticmethod
    def set_current_retry_count(context, counter_name_suffix, current_retry_count):
        counter_name_suffix = "current_retry_{0}".format(counter_name_suffix)
        context.setAttribute(counter_name_suffix, current_retry_count)

    def retry_or_fail(self, context, subject, max_retry_count, fail_message, wait_message):
        retry_count = self.get_current_retry_count(context, "{0}_stopped".format(subject))
        if retry_count > max_retry_count:
            raise RuntimeError("Reached maximum limit of {0} retries. {1}"
                               .format(max_retry_count, fail_message))
        else:
            print "{0} Done with retry {1}".format(wait_message, retry_count)
            self.increment_retry_counter(context, "{0}_stopped".format(subject))
            return "RETRY"

    def get_deployable_name(self, deployed_id=None):
        deployed_id = self.deployed.id if not deployed_id else deployed_id
        return deployed_id[deployed_id.rfind('/') + 1:]

    @staticmethod
    def check_connection(access_key, access_secret, verify_ssl=True):
        botocore_session = BotocoreSession()
        botocore_session.lazy_register_component('data_loader', lambda: commons.create_loader())

        session = Session(aws_access_key_id=access_key,
                          aws_secret_access_key=access_secret,
                          botocore_session=botocore_session)
        AWSHelper.set_ca_bundle_path()
        if not verify_ssl:
            client = session.client(service_name="s3", verify=False)
        else:
            client = session.client(service_name="s3")
        client.list_buckets()
        return True
