import os
import tempfile
from java.nio.file import Files, Paths, StandardCopyOption
import urllib
import com.xebialabs.deployit.plugin.aws.support.BotoLoader as BotoLoader
import commons
from boto3.session import Session
from botocore.session import Session as BotocoreSession
from botocore.config import Config

class AWSHelper(object):

    def __init__(self, deployed):
        self.deployed = deployed
        AWSHelper.__validate(deployed.container)
        self.session = AWSHelper.create_session(deployed.container)
        AWSHelper.set_ca_bundle_path()

    @staticmethod
    def create_session(container):
        session = AWSHelper.__get_session(container.useCredentials, container.accesskey, container.accessSecret)
        if container.roleName:
            response = AWSHelper.__get_assume_role_response(session, container)
            session = AWSHelper.__get_session(use_credentials=True,
                                              access_key=response['Credentials']['AccessKeyId'],
                                              access_secret=response['Credentials']['SecretAccessKey'],
                                              aws_session_token=response['Credentials']['SessionToken'])
        return session

    @staticmethod
    def __get_assume_role_response(session, container):
        sts_client = AWSHelper.__get_sts_client(session, container)
        assume_keywords = {
            'RoleArn': AWSHelper.__get_role_arn(container.roleName, container.accountId),
            'RoleSessionName': container.name
        }
        if container.externalId:
            assume_keywords['ExternalId'] = container.externalId

        response = sts_client.assume_role(**assume_keywords)
        assert response['Credentials'], 'Assume role failed, response: {0}'.format(response)
        return response

    @staticmethod
    def __get_role_arn(role_name, account_id):
        return 'arn:aws:iam::{0}:role/{1}'.format(account_id, role_name)

    @staticmethod
    def __get_sts_client(session, container):
        config = AWSHelper.create_config(container)
        if not container.verifySSL:
            return session.client('sts', verify=False, config=config)
        return session.client('sts', config=config)

    @staticmethod
    def __validate(container):
        if container.useCredentials:
            assert container.accesskey, "Access key should not be empty."
            assert container.accessSecret, "Access secret should not be empty."
        if container.roleName or container.accountId or container.externalId:
            assert container.accountId, "Assume Role's account id should not be empty when trying to assume role."
            assert container.roleName, "Assume Role's role name should not be empty when trying to assume role."

        if (container.proxyHost or container.proxyPort or container.proxyProtocol) and \
                not (container.proxyHost and container.proxyProtocol and container.proxyPort):
                raise Exception("Either all or none of proxy host, port and protocol should be provided.")

    @staticmethod
    def __get_session(use_credentials, access_key, access_secret, aws_session_token=None):
        botocore_session = BotocoreSession()
        botocore_session.lazy_register_component('data_loader',
                                                 lambda: commons.create_loader())
        session_keywords = {
            'botocore_session': botocore_session
        }
        if use_credentials:
            session_keywords['aws_access_key_id'] = access_key
            session_keywords['aws_secret_access_key'] = access_secret

        if aws_session_token:
            session_keywords['aws_session_token'] = aws_session_token

        return Session(**session_keywords)

    def get_aws_client(self, region, resource_name='ec2'):
        config = AWSHelper.create_config(self.deployed.container)
        if not self.deployed.container.verifySSL:
            return self.session.client(resource_name, region_name=region, verify=False, config=config)
        return self.session.client(resource_name, region_name=region, config=config)

    @staticmethod
    def create_config(container):
        config = None
        if container.proxyHost and container.proxyProtocol and container.proxyPort:
            proxy = "{}:{}".format(container.proxyHost, container.proxyPort)
            if container.proxyUser and container.proxyPassword:
                proxy = "{}:{}@{}".format(urllib.quote(container.proxyUser), urllib.quote(container.proxyPassword), proxy)
            proxy = "{}://{}".format(container.proxyProtocol, proxy)
            proxies = {'https': proxy, 'http': proxy}
            config = Config(proxies=proxies)

        return config

    @staticmethod
    def extract_file_from_jar(config_file):
        file_url = BotoLoader.getResourceBySelfClassLoader(config_file)
        if file_url:
            tmp_file, tmp_abs_path = tempfile.mkstemp()
            tmp_file.close()
            Files.copy(file_url.openStream(), Paths.get(tmp_abs_path), StandardCopyOption.REPLACE_EXISTING)
            return tmp_abs_path

        else:
            return None

    @staticmethod
    def set_ca_bundle_path():
        ca_bundle_path = AWSHelper.extract_file_from_jar("botocore/vendored/requests/cacert.pem")
        os.environ['REQUESTS_CA_BUNDLE'] = ca_bundle_path

    @staticmethod
    def is_starts_with_name(property_value):
        return property_value.lower().startswith('name:') if property_value else False

    @staticmethod
    def get_property_name(property_name):
        return property_name[5:]

    @staticmethod
    def is_success(response):
        return 299 >= response['ResponseMetadata']['HTTPStatusCode'] >= 200

    @staticmethod
    def remove_none_keys(dict):
        return {k: v for k, v in dict.iteritems() if v is not None}

    @staticmethod
    def remove_empty_and_none_values(dict):
        return {k: v for k, v in dict.iteritems() if (bool(v) if isinstance(v, (list, set)) else v is not None)}

    @staticmethod
    def get_current_retry_count(context, counter_name_suffix):
        counter_name_suffix = "current_retry_{0}".format(counter_name_suffix)
        current_retry_count = context.getAttribute(counter_name_suffix)
        current_retry_count = 1 if not current_retry_count else current_retry_count
        return current_retry_count

    @staticmethod
    def increment_retry_counter(context, counter_name_suffix):
        current_retry_count = AWSHelper.get_current_retry_count(context, counter_name_suffix)
        current_retry_count = current_retry_count + 1
        AWSHelper.set_current_retry_count(context, counter_name_suffix, current_retry_count)

    @staticmethod
    def set_current_retry_count(context, counter_name_suffix, current_retry_count):
        counter_name_suffix = "current_retry_{0}".format(counter_name_suffix)
        context.setAttribute(counter_name_suffix, current_retry_count)

    def retry_or_fail(self, context, subject, max_retry_count, fail_message, wait_message):
        retry_count = self.get_current_retry_count(context, "{0}_stopped".format(subject))
        if retry_count > max_retry_count:
            raise RuntimeError("Reached maximum limit of {0} retries. {1}"
                               .format(max_retry_count, fail_message))
        else:
            print "{0} Done with retry {1}".format(wait_message, retry_count)
            self.increment_retry_counter(context, "{0}_stopped".format(subject))
            return "RETRY"

    def get_deployable_name(self, deployed_id=None):
        deployed_id = self.deployed.id if not deployed_id else deployed_id
        return deployed_id[deployed_id.rfind('/') + 1:]

    @staticmethod
    def check_connection(container):
        AWSHelper.__validate(container)
        session = AWSHelper.__get_session(container.useCredentials, container.accesskey, container.accessSecret)
        print 'Verifying connection.'
        AWSHelper.set_ca_bundle_path()
        client = AWSHelper.__get_sts_client(session, container)
        client.get_caller_identity()
        if container.roleName:
            print 'Trying to assume role {0}.'.format(AWSHelper.__get_role_arn(container.roleName, container.accountId))
            AWSHelper.__get_assume_role_response(session, container)

        return True
