from cloudformation.utils.resource_mapper import ResourceMapper
import botocore


class S3BucketMapper(ResourceMapper):
    def __init__(self, client):
        super(S3BucketMapper, self).__init__(["aws.s3.Bucket"])
        self.s3_client = client

    def create_cis(self, cf_resource, cloud):
        if self.types_supported(cf_resource['PhysicalResourceId']):
            cloud_id = cloud.id
            bucket_id = cf_resource['PhysicalResourceId']
            print("Creating CI of type 'aws.s3.Bucket' from '%s'" % bucket_id)
            region = self.s3_client.get_bucket_location(Bucket = bucket_id)['LocationConstraint']
            id = cloud_id + "/" + bucket_id + "-S3Bucket"
            bucket_website = self.get_website(bucket_id)
            bucket_logging = self.get_logging(bucket_id)
            properties = {'bucketName':           bucket_id, 
                          'region':               region, 
                          'policy':               self.get_policy(bucket_id), 
                          'logging':              bucket_logging['logging'], 
                          'targetBucket':         bucket_logging['targetBucket'], 
                          'targetPrefix':         bucket_logging['targetPrefix'], 
                          'endpoint':             "%s.s3-website-%s.amazonaws.com" % (bucket_id, region), 
                          'staticWebsiteHosting': bucket_website['staticWebsiteHosting'], 
                          'indexDocument':        bucket_website['indexDocument'], 
                          'errorDocument':        bucket_website['errorDocument'], 
                          }
            return [
                super(S3BucketMapper, self)._create_ci("aws.s3.Bucket", id, properties)
                ]
        else:
            return None

    def get_policy(self, bucket_id):
        try:
            return self.s3_client.get_bucket_policy(
                Bucket = bucket_id
            )['Policy']
        except botocore.exceptions.ClientError as e:
            if e.response['Error']['Code'] == 'NoSuchBucketPolicy':
                print("The S3 Bucket has no policy.")
            else:
                raise e
        return ""

    def get_logging(self, bucket_id):
        logging = self.s3_client.get_bucket_logging(
            Bucket = bucket_id
        )
        if 'LoggingEnabled' in logging:
            return {
                'logging':      True, 
                'targetBucket': logging['LoggingEnabled']['TargetBucket'], 
                'targetPrefix': logging['LoggingEnabled']['TargetPrefix']
            }
        else:
            print("The S3 Bucket has no logging enabled.")
            return {
                'logging':      False, 
                'targetBucket': "", 
                'targetPrefix': ""
            }

    def get_website(self, bucket_id):
        try:
            website =  self.s3_client.get_bucket_website(
                Bucket = bucket_id
            )
            return {
                'staticWebsiteHosting': True, 
                'indexDocument':        website['IndexDocument']['Suffix'], 
                'errorDocument':        website['ErrorDocument']['Key']
            }
        except botocore.exceptions.ClientError as e:
            if e.response['Error']['Code'] == 'NoSuchWebsiteConfiguration':
                print("The S3 Bucket has no website configuration.")
            else:
                raise e
        return {
            'staticWebsiteHosting': False, 
            'indexDocument':        "", 
            'errorDocument':        ""
        }
