from org.cloudfoundry.operations import DefaultCloudFoundryOperations
from org.cloudfoundry.operations.applications import DeleteApplicationRequest, PushApplicationRequest, \
    StartApplicationRequest, StopApplicationRequest, GetApplicationRequest, RenameApplicationRequest, \
    GetApplicationManifestRequest, ApplicationManifest, ApplicationManifestUtils
from org.cloudfoundry.operations.spaces import CreateSpaceRequest, DeleteSpaceRequest, RenameSpaceRequest
from org.cloudfoundry.operations.services import BindServiceInstanceRequest, UnbindServiceInstanceRequest
from org.cloudfoundry.reactor import DefaultConnectionContext
from org.cloudfoundry.reactor.client import ReactorCloudFoundryClient
from org.cloudfoundry.reactor.uaa import ReactorUaaClient
from org.cloudfoundry.reactor.tokenprovider import PasswordGrantTokenProvider
from org.cloudfoundry.client.v2.applications import ListApplicationRoutesRequest, ListApplicationServiceBindingsRequest
from org.cloudfoundry.client.v2.routes import DeleteRouteRequest

from org.cloudfoundry.operations.applications import ApplicationManifest, ApplicationManifestUtils, Docker, PushApplicationManifestRequest

from java.io import File


class OrgClient(object):
    def __init__(self, api_endpoint, username, password, organization_name, ignore_ssl=False):
        connection_context = DefaultConnectionContext.builder().apiHost(api_endpoint).skipSslValidation(
            ignore_ssl).build()
        token_provider = PasswordGrantTokenProvider.builder().password(password).username(username).build()
        self.rest_client = ReactorCloudFoundryClient.builder().connectionContext(connection_context).tokenProvider(
            token_provider).build()
        self._apiEndpoint = api_endpoint
        self._uaa_client = ReactorUaaClient.builder().connectionContext(connection_context).tokenProvider(
            token_provider).build()
        self._client = DefaultCloudFoundryOperations.builder().cloudFoundryClient(self.rest_client).uaaClient(
            self._uaa_client).organization(organization_name).build()
        org_names = [organization.name for organization in self._client.organizations().list().toIterable()]
        if organization_name not in org_names:
            raise RuntimeError("Organization with name {} not found on server.".format(organization_name))
        self._org = organization_name

    @classmethod
    def create_client(cls, container):
        return cls(container.apiEndpoint, container.username, container.password, container.organizationName,
                   ignore_ssl=container.ignoreSsl)

    @classmethod
    def create_discovery_client(cls, container):
        return cls(container.getProperty("apiEndpoint"), container.getProperty("username"), container.getProperty("password"), container.getProperty("organizationName"),
                   ignore_ssl=container.getProperty("ignoreSsl"))

    def create_space(self, space_name):
        print("Making request to create space %s" % space_name)
        self._client.spaces().create(
            CreateSpaceRequest.builder().name(space_name).organization(self._org).build()).block()

    def delete_space(self, space_name):
        print("Making request to delete space %s" % space_name)
        self._client.spaces().delete(DeleteSpaceRequest.builder().name(space_name).build()).block()

    def rename_space(self, space_name, new_space_name):
        print ("Making request to rename space %s to %s" % (space_name, new_space_name))
        self._client.spaces().rename(
            RenameSpaceRequest.builder().name(space_name).newName(new_space_name).build()).block()

    def discover_spaces(self):
        spaces = []
        for space in self._client.spaces().list().toIterable():
            spaces.append(space)
        return spaces



class SpaceClient(OrgClient):
    def __init__(self, api_endpoint, username, password, organization_name, space_name, ignore_ssl=False):
        super(SpaceClient, self).__init__(api_endpoint, username, password, organization_name, ignore_ssl)
        self._space_client = DefaultCloudFoundryOperations.builder().cloudFoundryClient(self.rest_client). \
            uaaClient(self._uaa_client).organization(self._org).space(space_name).build()

    @classmethod
    def create_client(cls, container):
        org = container.container
        return cls(org.apiEndpoint, org.username, org.password, org.organizationName,
                   container.spaceName if container.spaceName else container.name, ignore_ssl=org.ignoreSsl)

    @classmethod
    def create_space_discovery_client(cls, org, space):
        return cls(org.getProperty("apiEndpoint"), org.getProperty("username"), org.getProperty("password"), org.getProperty("organizationName"),
                   space.name, ignore_ssl=org.getProperty("ignoreSsl"))

    def get_client(self):
        return self._space_client

    def create_application(self, app_name, file, memory=512, instances=1, build_pack=None, hostname=None,
                           random_route=None, no_route=False):
        print("Making request to create application %s" % app_name)
        if self.application_exists(app_name):
            raise RuntimeError("App with name {} already exists".format(app_name))
        self.push_application(app_name, file, memory, instances, build_pack, hostname, random_route, no_route)

    def push_application(self, app_name, file, memory=512, instances=1, build_pack=None, hostname=None,
                         random_route=None, no_route=False):
        print("Making request to push application %s" % app_name)
        request_builder = PushApplicationRequest.builder().name(app_name).application(File(file).toPath()).buildpack(
            build_pack) \
            .instances(instances).memory(memory).randomRoute(random_route).noRoute(no_route).host(hostname)
        self._space_client.applications().push(request_builder.noStart(True).build()).block()

    def delete_application(self, app_name):
        print("Making request to delete application %s" % app_name)
        if self.application_exists(app_name):
            self._space_client.applications().delete(
                DeleteApplicationRequest.builder().name(app_name).deleteRoutes(True).build()).block()

    def application_exists(self, app_name):
        print("Making request to check if application name %s already exists" % app_name)
        for application in self._space_client.applications().list().toIterable():
            if application.getName() == app_name:
                return True
        return False

    def get_application_manifest(self, app_name):
        return self._space_client.applications().getApplicationManifest(GetApplicationManifestRequest.builder().name(app_name).build()).block()

    def discover_applications(self):
        applications = []
        for app in self._space_client.applications().list().toIterable():
            applications.append(app)
        return applications

    def delete_routes(self, app_id):
        routes = self.rest_client.applicationsV2().listRoutes(
            ListApplicationRoutesRequest.builder().applicationId(app_id).build()).block().resources
        for route in routes:
            print "Deleting route: {} with host: {}".format(route.metadata.id, route.entity.host)
            self.rest_client.routes().delete(DeleteRouteRequest.builder().routeId(route.metadata.id).build()).block()

    def get_application_details(self, app_name):
        print("Making request to get application details about %s" % app_name)
        return self._space_client.applications().get(GetApplicationRequest.builder().name(app_name).build()).block()

    def start_application(self, app_name):
        print("Making request to start application %s" % app_name)
        self._space_client.applications().start(StartApplicationRequest.builder().name(app_name).build()).block()

    def stop_application(self, app_name):
        print("Making request to stop application %s" % app_name)
        self._space_client.applications().stop(StopApplicationRequest.builder().name(app_name).build()).block()

    def rename_application(self, app_name, new_app_name):
        print("Making request to rename application %s to %s" % (app_name, new_app_name))
        self._space_client.applications().rename(
            RenameApplicationRequest.builder().name(app_name).newName(new_app_name).build()).block()

    def bind_app_to_service(self, application_name, service_instance_name, parameters):
        print("Making request to bind service %s to application %s" % (service_instance_name, application_name))
        self._space_client.services().bind(
            BindServiceInstanceRequest.builder().serviceInstanceName(service_instance_name).applicationName(
                application_name).parameters(parameters).build()).block()

    def unbind_app_from_service(self, application_name, service_instance_name):
        print("Making request to unbind service %s from application %s" % (service_instance_name, application_name))
        self._space_client.services().unbind(
            UnbindServiceInstanceRequest.builder().serviceInstanceName(service_instance_name).applicationName(
                application_name).build()).block()

    def check_applications_exists(self, app_names):
        for app_name in app_names:
            if self.application_exists(app_name):
                raise RuntimeError("Application with name {} already exists".format(app_name))

    def  push_manifest_application(self, file):
        for app in ApplicationManifestUtils.read(file):
            application_manifest_builder = ApplicationManifest.builder().from(app)
            print("Making request to push application from manifest: \n" + str(app))
            self._space_client.applications().pushManifest(PushApplicationManifestRequest.builder().manifest(application_manifest_builder.build()).build()).block()
